#include    "sched.h"

/**
 * Search for a process which is in main memory, but which is not the scheduler
 * or locked (i.e. already being swapped out), and whose state is @ref SWAIT
 * or @ref SSTOP (but not @ref SSLEEP) (i.e. the process is waiting for an
 * event of low precedence, or has stopped during tracing (see Chapter
 * Thirteen)).
 *
 * @return Process or NULL if none found
 *
 * Note that there seems to be a bias here against processes whose ?proc?
 * entries are early in the @ref proc array.
 */
struct Proc* getEasyCore()
{
  struct Proc      *itr;

  for (itr = &proc[0]; itr < &proc[NPROC]; itr++)
  {
    if( (flagTest( itr->p_flag, SSYS ) == 0) &&
      (flagTest( itr->p_flag, SLOCK ) == 0) &&
      (flagTest( itr->p_flag, SLOAD ) > 0) &&
      (itr->p_stat == SWAIT || itr->p_stat == SSTOP))
    {
      return    itr;
    }
  }

  return    NULL;
}


/**
 * Search for the process which is loaded, but is not the scheduler or locked,
 * whose state is @ref SRUN  or @ref SSLEEP (i.e. ready to run, or waiting for
 * an event of high precedence) and which has been in main memory for the
 * longest time;
 *
 * @return the process or NULL if none found
 */
struct Proc* getOldesSwappableProc()
{
  struct Proc      *itr;
  struct Proc      *oldest = NULL;
  int       age;

  age = -1;                                       // @line:2005
  for (itr = &proc[0]; itr < &proc[NPROC]; itr++)
  {
    if( (flagTest( itr->p_flag, SSYS ) == 0) &&
      (flagTest( itr->p_flag, SLOCK ) == 0) &&
      (flagTest( itr->p_flag, SLOAD ) > 0) &&
      (itr->p_stat == SRUN || itr->p_stat == SSLEEP) &&
      (itr->p_time > age) )
    {
      oldest = itr;
      age    = itr->p_time;
    }
  }

  return    oldest;
}


/**
 * Searches for a process who can be swapped out and swaps it out.
 *
 * @param dontSwapRunningProc true when a running process can't be swapped out
 * @return @ref RUNIN or @ref RESCHEDULE
 */
SchedulerState swapOut( int dontSwapRunningProc )
{
  struct Proc      *swpP;

  spl6 ();
  swpP = getEasyCore();

  if( swpP == NULL )
  {
    if( dontSwapRunningProc )
    {
      return  RUNIN;
    }

    swpP    = getOldesSwappableProc();

    if( swpP == NULL )
    {
      return    RUNIN;
    }
    else if( swpP->p_time < 2 )                   // @line:2013
    {
      return  RUNIN;
    }
  }

// swap user out
  spl0 ();                                        // @line:2022
  flagClear( swpP->p_flag, SLOAD );
  xswap (swpP, 1, 0);

  return    RESCHEDULE;
}


/**
 * Swaps the text segment in.
 *
 * panic: swap error -- IO error while swapping.
 *  this is the one panic that should be
 *  handled in a less drastic way. Its
 *  very hard.
 *
 * @param txt Text segment to be swapped in
 * @param addr Address where the text segment is copied at
 * @return address offset
 */
int swapTextIn( struct Text* txt, void *addr )
{
  int   offset  = 0;

  if( txt != NULL )
  {
    if (txt->x_ccount == 0)
    {
      if (swap (txt->x_daddr, addr, txt->x_size, B_READ))
      {
        panic ("swap error");
      }
      txt->x_caddr = addr;
      offset = txt->x_size;
    }
    txt->x_ccount++;
  }

  return    offset;
}


/**
 * Swap the process swpProc in.
 *
 * panic: swap error -- IO error while swapping.
 *  this is the one panic that should be
 *  handled in a less drastic way. Its
 *  very hard.
 *
 * @param swpProc Process to be swapped in
 * @param addr Address where the process is copied at
 * @return @ref RESCHEDULE
 */
SchedulerState swapIn( struct Proc *swpProc, void *addr )
{
  int   offset;

  offset    = swapTextIn( swpProc->p_textp, addr );

  addr  = addr + offset;

// @line:2042
  if (swap (swpProc->p_addr, addr, swpProc->p_size, B_READ))
  {
    panic ("swap error");
  }

// @line:2044
  mfree (swapmap, (swpProc->p_size + 7) / 8, swpProc->p_addr);
  swpProc->p_addr = addr;
  flagSet( swpProc->p_flag, SLOAD );
  swpProc->p_time = 0;

  return    RESCHEDULE;
}


/**
 * Find oldest process who is ready to be swapped in.
 *
 * @return Process or NULL if none found
 */
struct Proc* getOldestToSwpIn()
{
  struct Proc      *oldest;
  struct Proc      *itr;
  int       age;

  oldest = NULL;
  age    = -1;

  for (itr = &proc[0]; itr < &proc[NPROC]; itr++)
  {
    if( (itr->p_stat == SRUN) &&
      (flagTest( itr->p_flag, SLOAD ) == 0) &&
      (itr->p_time > age) )
    {
      oldest = itr;
      age    = itr->p_time;
    }
  }

  return    oldest;
}


/**
 * Returns the memory size of the process p, consisting of the data and text area.
 *
 * @return Size for the process
 */
int getSizeForProc( struct Proc *p )
{
  struct Text  *t;
  int   size;

  size = p->p_size;

  t    = p->p_textp;

  if( t != NULL )
  {
    if( t->x_ccount == 0 )
    {
      size = size + t->x_size;
    }
  }

  return    size;
}


/**
 * The main swap-scheduling function. Swappes processes in or out.
 *
 * @return @ref RUNOUT, @ref RUNIN or @ref RESCHEDULE
 */
SchedulerState schedule()
{
  struct Proc   *swpP;
  int           size;
  void          *addr;

  spl6 ();                                        // @line:1958

  swpP    = getOldestToSwpIn();

  if( swpP == NULL )                              // @line:1966
  {
    return  RUNOUT;
  }

  spl0 ();                                        // @line:1976

  size  = getSizeForProc( swpP );

  addr = malloc( coremap, size );

  if( addr != NULL )                              // @line:1982
  {
    return  swapIn( swpP, addr );
  }
  else
  {                                               // none found
    return  swapOut( swpP->p_time < 3 );          // If the image to be swapped in has been out less than 3 seconds, then situation B holds;
  }
}


/**
 * The main loop of the scheduling (swapping)
 * process.
 * The basic idea is:
 *  see if anyone wants to be swapped in;
 *  swap out processes until there is room;
 *  swap him in;
 *  repeat.
 * Although it is not remarkably evident, the basic
 * synchronization here is on the runin flag, which is
 * slept on and is set once per second by the clock routine.
 * Core shuffling therefore takes place once per second.
 */
void sched ()                                     // @line:1940
{
  SchedulerState    state;

  while( 1 )
  {
    do
    {
      state   = schedule();
    } while( state == RESCHEDULE );

    switch( state )
    {
      case RUNOUT:
      {
        runout++;
        sleep (&runout, PSWP);
        break;
      }

      case RUNIN:
      {
        runin++;
        sleep (&runin, PSWP);
        break;
      }
    }
  }
}
