/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package ast.traverser;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import ast.Ast;
import ast.expression.Expression;
import ast.statement.AssignmentStmt;
import ast.statement.CaseStmt;
import ast.statement.DoWhile;
import ast.statement.IfStmt;
import ast.statement.Statement;
import ast.statement.WhileStmt;
import ast.variable.Variable;
import ast.variable.VariableRefLinked;
import ast.variable.VariableRefUnlinked;

/*
 * replaces code like
 *       tmp4_t = tmp6;
 *       tmp6_t = (tmp7 + tmp6);
 *       tmp7_t = tmp6;
 *
 * with this
 *       tmp4_t = tmp6;
 *       tmp6_t = (tmp7 + tmp4_t);
 *       tmp7_t = tmp4_t;
 * 
 * i.e. uses the last assigned variable with the same value
 * 
 */
public class LiveRangeShortener extends AstTraverser<Map<Variable, Variable>> {

  static public void process(Ast ast) {
    LiveRangeShortener collector = new LiveRangeShortener();
    collector.visit(ast, new HashMap<Variable, Variable>());
  }

  public LiveRangeShortener() {
    super(new StmtLiveRangeShortener());
  }

}

class StmtLiveRangeShortener extends AstStatementTraverser<Map<Variable, Variable>> {
  private ExprLiveRangeShortener exprvisitor = new ExprLiveRangeShortener();

  @Override
  public Expression visit(Expression expr, Map<Variable, Variable> param) {
    return exprvisitor.visit(expr, param);
  }

  @Override
  public Variable visit(Variable expr, Map<Variable, Variable> param) {
    return expr;
  }

  @Override
  protected Statement visitAssignmentStmt(AssignmentStmt obj, Map<Variable, Variable> param) {
    obj.setSource(visit(obj.getSource(), param));
    if (obj.getDestination() instanceof VariableRefLinked) {
      VariableRefLinked dst = (VariableRefLinked) obj.getDestination();
      param.remove(dst.getReference());
      if (obj.getSource() instanceof VariableRefLinked) {
        VariableRefLinked src = (VariableRefLinked) obj.getSource();
        if (!src.getReference().equals(dst.getReference())) {
          param.put(src.getReference(), dst.getReference());
        }
      }
    }
    return obj;
  }

  @Override
  protected Statement visitIfStmt(IfStmt obj, Map<Variable, Variable> param) {
    obj.setCondition(visit(obj.getCondition(), param));
    HashMap<Variable, Variable> tb = new HashMap<Variable, Variable>(param);
    HashMap<Variable, Variable> eb = new HashMap<Variable, Variable>(param);
    obj.setThenBranch(visit(obj.getThenBranch(), tb));
    obj.setElseBranch(visit(obj.getElseBranch(), eb));

    merge(param, tb, eb);

    return obj;
  }

  private void merge(Map<Variable, Variable> ret, Map<Variable, Variable> in1, Map<Variable, Variable> in2) {
    ret.clear();
    Set<Variable> common = new HashSet<Variable>(in1.keySet());
    common.retainAll(in2.keySet());
    for (Variable var : common) {
      if (in1.get(var).equals(in2.get(var))) {
        ret.put(var, in1.get(var));
      }
    }
  }

  @Override
  protected Statement visitWhileStmt(WhileStmt obj, Map<Variable, Variable> param) {
    HashMap<Variable, Variable> bp = new HashMap<Variable, Variable>();
    obj.setBody(visit(obj.getBody(), bp));
    merge(param, bp, new HashMap<Variable, Variable>(param));
    obj.setCondition(visit(obj.getCondition(), param));
    return obj;
  }

  @Override
  protected Statement visitDoWhileStmt(DoWhile obj, Map<Variable, Variable> param) {
    param.clear();
    obj.setBody(visit(obj.getBody(), param));
    obj.setCondition(visit(obj.getCondition(), param));
    return obj;
  }

  @Override
  protected Statement visitCaseStmt(CaseStmt obj, Map<Variable, Variable> param) {
    obj.setCondition(visit(obj.getCondition(), param));

    visit(obj.getOther(), new HashMap<Variable, Variable>(param));
    for (Statement stmt : obj.getOption().values()) {
      visit(stmt, new HashMap<Variable, Variable>(param));
    }
    param.clear(); // TODO replace with merge
    return obj;
  }

}

class ExprLiveRangeShortener extends AstExpressionTraverser<Map<Variable, Variable>> {

  @Override
  protected Expression visitVariableRefLinked(VariableRefLinked obj, Map<Variable, Variable> param) {
    Variable var = obj.getReference();
    while (param.containsKey(var)) {
      var = param.get(var);
    }
    return new VariableRefLinked(var);
  }

  @Override
  protected Expression visitVariableRefUnlinked(VariableRefUnlinked obj, Map<Variable, Variable> param) {
    throw new RuntimeException("Not yet implemented");
  }

}
