/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package ast.type;

/*
 * Integer  Pointer
 *   |        |
 *   +---+----+
 *       |
 *     Number   Boolean
 *       |         |
 *       +----+----+
 *            |
 *         Scalar  String  Void
 *            |       |     |
 *            +------++-----+
 *                   |
 *                Generic
 */

public enum Type {
  Generic, Integer, Boolean, Number, Pointer, String, Void, Scalar;

  public Type getSupertype() {
    switch (this) {
      case Generic:
        return Generic;
      case Scalar:
        return Generic;
      case Boolean:
        return Scalar;
      case Number:
        return Scalar;
      case Integer:
        return Number;
      case Pointer:
        return Number;
      case String:
        return Generic;
      case Void:
        return Generic;
      default:
        throw new RuntimeException("Not yet handled type: " + this);
    }
  }

  public boolean isSupertypeOf(Type other) {
    while ((this != other) && (other != Generic)) {
      other = other.getSupertype();
    }
    return this == other;
  }

  public int getDeepth() {
    int deepth = 0;
    for (Type itr = this; itr != Generic; itr = itr.getSupertype()) {
      deepth++;
    }
    return deepth;
  }

  static public Type getCommonType(Type a, Type b) {
    int da = a.getDeepth();
    int db = b.getDeepth();

    if (da != db) {
      for (int i = da; i > db; i--) {
        a = a.getSupertype();
      }
      for (int i = db; i > da; i--) {
        b = b.getSupertype();
      }
    }

    assert (a.getDeepth() == b.getDeepth());

    while (a != b) {
      a = a.getSupertype();
      b = b.getSupertype();
    }

    return a;
  }

  public static Type getNarrowType(Type a, Type b) {
    assert (a.isSupertypeOf(b) || b.isSupertypeOf(a));
    if (a.getDeepth() > b.getDeepth()) {
      return a;
    } else {
      return b;
    }
  }

}
