/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.function;


import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import org.jgrapht.DirectedGraph;
import org.jgrapht.EdgeFactory;

import util.NumPrint;
import cfg.basicblock.BasicBlock;
import cfg.basicblock.BbEdge;
import cfg.statement.JumpStmt;
import cfg.statement.Statement;

public class FunctionGraph implements DirectedGraph<BasicBlock, BbEdge> {
  private HashMap<Long, BasicBlock> bbs = new HashMap<Long, BasicBlock>();

  public FunctionGraph(Collection<BasicBlock> bbs) {
    for (BasicBlock bb : bbs) {
      this.bbs.put(bb.getId(), bb);
    }
    updateEdges();
  }

  public void updateEdges() {
    for (BasicBlock bb : bbs.values()) {
      bb.setOutlist(new HashSet<BbEdge>());
      bb.setInlist(new HashSet<BbEdge>());
    }
    for (BasicBlock bb : bbs.values()) {
      linkOutEdges(bb);
    }
  }

  private void linkOutEdges(BasicBlock bb) {
    Statement last = bb.getCode().get(bb.getCode().size() - 1);

    if (last instanceof JumpStmt) {
      JumpStmt jmp = (JumpStmt) last;
      for (int i = 0; i < jmp.getJmpDst().size(); i++) {
        long dstaddr = jmp.getJmpDst().get(i);
        if (!bbs.containsKey(dstaddr)) {
          // if you debugging to this place, check for tail call optimization
          throw new RuntimeException("Basic block not found: " + NumPrint.toString(dstaddr) + " (" + bb + ")");
        }
        addEdge(bb, bbs.get(dstaddr), i);
      }
    } else {
      // assert ( last instanceof RetStmt); // can also be a call
    }
  }

  public BasicBlock getVertex(long id) {
    return bbs.get(id);
  }

  private void addEdge(BasicBlock src, BasicBlock dst, int nr) {
    assert (src != null);
    assert (dst != null);
    new BbEdge(src, dst, nr);
  }

  public BbEdge addEdge(BasicBlock src, BasicBlock dst) {
    throw new RuntimeException("Not yet implemented");
  }

  public boolean addEdge(BasicBlock src, BasicBlock dst, BbEdge e) {
    assert (e.getSrc() == src);
    assert (e.getDst() == dst);
    assert (bbs.containsValue(src));
    assert (bbs.containsValue(dst));
    // assert( getEdge(src, dst) == null );
    JumpStmt jmp = (JumpStmt) src.getCode().getLast();
    assert (jmp.getJmpDst().contains(dst.getId()));
    return true;
  }

  public boolean addVertex(BasicBlock bb) {
    assert (!bbs.containsKey(bb.getId()));
    this.bbs.put(bb.getId(), bb);
    return true;
  }

  public boolean containsEdge(BbEdge arg0) {
    throw new RuntimeException("Not yet implemented");
  }

  public boolean containsEdge(BasicBlock arg0, BasicBlock arg1) {
    assert (bbs.containsKey(arg0.getId()));
    assert (bbs.containsKey(arg1.getId()));
    for (BbEdge e : arg0.getOutlist()) {
      if (e.getDst() == arg1) {
        return true;
      }
    }
    return false;
  }


  public boolean containsVertex(BasicBlock arg0) {
    throw new RuntimeException("Not yet implemented");
  }

  public Set<BbEdge> edgeSet() {
    HashSet<BbEdge> res = new HashSet<BbEdge>();
    for (BasicBlock bb : bbs.values()) {
      res.addAll(bb.getOutlist());
    }
    return res;
  }


  public Set<BbEdge> edgesOf(BasicBlock arg0) {
    throw new RuntimeException("Not yet implemented");
  }


  public Set<BbEdge> getAllEdges(BasicBlock arg0, BasicBlock arg1) {
    throw new RuntimeException("Not yet implemented");
  }


  public BbEdge getEdge(BasicBlock src, BasicBlock dst) {
    assert (bbs.containsValue(src));
    assert (bbs.containsValue(dst));
    for (BbEdge e : src.getOutlist()) {
      if (e.getDst() == dst) {
        return e;
      }
    }
    return null;
  }


  public EdgeFactory<BasicBlock, BbEdge> getEdgeFactory() {
    throw new RuntimeException("Not yet implemented");
  }


  public BasicBlock getEdgeSource(BbEdge arg0) {
    return arg0.getSrc();
  }


  public BasicBlock getEdgeTarget(BbEdge arg0) {
    return arg0.getDst();
  }


  public double getEdgeWeight(BbEdge arg0) {
    throw new RuntimeException("Not yet implemented");
  }


  public boolean removeAllEdges(Collection<? extends BbEdge> arg0) {
    for (BbEdge edge : arg0) {
      removeEdge(edge);
    }
    return true;
  }


  public Set<BbEdge> removeAllEdges(BasicBlock arg0, BasicBlock arg1) {
    throw new RuntimeException("Not yet implemented");
  }


  public boolean removeAllVertices(Collection<? extends BasicBlock> vertices) {
    for (BasicBlock v : vertices) {
      removeVertex(v);
    }
    return true;
  }


  public boolean removeEdge(BbEdge arg0) {
    throw new RuntimeException("Not yet implemented");
  }


  public BbEdge removeEdge(BasicBlock arg0, BasicBlock arg1) {
    throw new RuntimeException("Not yet implemented");
  }


  public boolean removeVertex(BasicBlock arg0) {
    assert (bbs.containsKey(arg0.getId()));
    bbs.remove(arg0.getId());
    return true;
  }


  public Set<BasicBlock> vertexSet() {
    return new HashSet<BasicBlock>(bbs.values());
  }


  public int inDegreeOf(BasicBlock arg0) {
    return arg0.getInlist().size();
  }


  public Set<BbEdge> incomingEdgesOf(BasicBlock arg0) {
    return arg0.getInlist();
  }


  public int outDegreeOf(BasicBlock arg0) {
    throw new RuntimeException("Not yet implemented");
  }


  public Set<BbEdge> outgoingEdgesOf(BasicBlock arg0) {
    return arg0.getOutlist();
  }

}
