/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.function.argument;


import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import cfg.Assignable;
import cfg.IrTraverser;
import cfg.IrType;
import cfg.expression.CallExprLinked;
import cfg.expression.CallExprPointer;
import cfg.expression.CallExprUnlinked;
import cfg.function.Function;
import cfg.statement.AssignmentStmt;
import cfg.variable.SsaVariable;
import cfg.variable.Variable;
import cfg.variable.VariableName;

//TODO what is with the function foo(a){return a;), called via function pointer. a might be not written (killed) and hence not detected?

/**
 * Adds definition of return values to the function call site.
 * 
 * For this, all return values of a called function are appended to the left hand side of an assignment.
 * 
 * @author urs
 * 
 */
public class FuncReturnAppend extends IrTraverser<Collection<VariableName>, Void> {

  public static void process(Object elem) {
    FuncReturnAppend fra = new FuncReturnAppend();
    fra.visit(elem, null);
  }

  // adds variables returned by a function to the destination of the assignment
  @Override
  protected Collection<VariableName> visitAssignmentStmt(AssignmentStmt obj, Void param) {
    Collection<VariableName> ret = visit(obj.getSource(), null);
    if (ret != null) {
      for (VariableName var : ret) {
        if (!contains(obj.getDestination(), var)) {
          obj.getDestination().add(new SsaVariable(var, obj.getNumber()));
        }
      }
    }
    return null;
  }

  private boolean contains(List<Assignable> destination, VariableName var) {
    for (Assignable itm : destination) {
      if ((itm.getIrType() == IrType.Variable)) {
        if (((Variable) itm).getName() == var) {
          return true;
        }
      }
    }
    return false;
  }

  @Override
  protected Collection<VariableName> visitCallExprLinked(CallExprLinked obj, Void param) {
    return obj.getFunc().getReturns();
  }

  @Override
  protected Collection<VariableName> visitCallExprUnlinked(CallExprUnlinked obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  /*
   * Only variables returned by all possible functions can be the return values
   * not really tested
   */
  @Override
  protected Collection<VariableName> visitCallExprPointer(CallExprPointer obj, Void param) {
    if (obj.getCandidates().isEmpty()) {
      return new ArrayList<VariableName>();
    }
    ArrayList<Collection<VariableName>> names = new ArrayList<Collection<VariableName>>();
    for (Function func : obj.getCandidates().values()) {
      names.add(func.getReturns());
    }
    HashSet<VariableName> res = new HashSet<VariableName>(names.get(0));
    for (int i = 1; i < names.size(); i++) {
      res.retainAll(names.get(i));
    }
    return res;
  }
}
