/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.function.library.stdio;


import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import knowledge.KnowOwner;
import cfg.IntConstant;
import cfg.IrType;
import cfg.basicblock.BasicBlock;
import cfg.expression.CallExprLinked;
import cfg.expression.Expression;
import cfg.expression.VariableRefLinked;
import cfg.statement.AssignmentStmt;
import cfg.statement.Statement;
import cfg.variable.ArrayAccess;

public class StdioStringParser {
  static public List<FormatToken> parseFormat(String format) {
    ArrayList<FormatToken> res = new ArrayList<FormatToken>();
    String running = "";
    int i = 0;
    while (i < format.length()) {
      switch (format.charAt(i)) {
        case '%': {
          if (running.length() > 0) {
            res.add(new StringFormat(running));
            running = "";
          }
          i++;
          if (format.charAt(i) != 'i') {
            throw new RuntimeException("format to complicated for me: " + format);
          }
          res.add(new IntegerFormat());
          break;
        }
        case 10: {
          if (running.length() > 0) {
            res.add(new StringFormat(running));
            running = "";
          }
          res.add(new NewlineFormat());
          break;
        }
        default: {
          running += format.charAt(i);
          break;
        }
      }
      i++;
    }
    if (running.length() > 0) {
      res.add(new StringFormat(running));
    }
    return res;
  }

  static public int getArgCount(List<FormatToken> parsed) {
    int argcount = 0;
    for (FormatToken fmt : parsed) {
      switch (fmt.getType()) {
        case Integer: {
          argcount++;
          break;
        }
      }
    }
    return argcount;
  }

  static public long getFormatStringAddr(CallExprLinked obj, KnowOwner ko) {
    assert (obj.getParam().size() >= 1);
    Expression arg0 = null;

    Statement stmt = ko.getExprOwner(obj);
    BasicBlock bb = ko.getStmtOwner(stmt);

    arg0 = getArrayWriteExpr(bb.getCode(), bb.getCode().indexOf(stmt) - 1, 0);
    assert (arg0 != null);

    long addr = followAssignements(arg0, ko);
    return addr;
  }

  private static long followAssignements(Expression expr, KnowOwner ko) {
    switch (expr.getIrType()) {
      case IntConstant: {
        return ((IntConstant) expr).getValue();
      }
      case VariableRefLinked: {
        Statement stmt = ko.getVarOwner(((VariableRefLinked) expr).getReference());
        if (stmt instanceof AssignmentStmt) {
          return followAssignements(((AssignmentStmt) stmt).getSource(), ko);
        } else {
          throw new RuntimeException("expected assignement: " + stmt);
        }
      }
      default: {
        throw new RuntimeException("unhandled expression type: " + ko.getExprOwner(expr));
      }
    }
  }

  public static Expression getArrayWriteExpr(LinkedList<Statement> code, int start, int index) {
    int idx = start;
    while (idx >= 0) {
      Statement stmt = code.get(idx);
      if (stmt.getIrType() == IrType.AssignmentStmt) {
        AssignmentStmt ass = (AssignmentStmt) stmt;
        if (ass.getDestination().size() == 1) {
          if (ass.getDestination().get(0) instanceof ArrayAccess) {
            ArrayAccess array = (ArrayAccess) ass.getDestination().get(0);
            assert (array.getIndex() instanceof IntConstant);
            assert (((IntConstant) array.getIndex()).getValue() == 0);
            if (array.getBase().getOffset() == index) {
              return ass.getSource();
            }
          }
        }
      }
      idx--;
    }
    return null;
  }

}
