/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.matcher;

import cfg.IntConstant;
import cfg.IrItr;
import cfg.IrType;
import cfg.expression.CallExprLinked;
import cfg.expression.IntegerExpr;
import cfg.expression.IntegerOp;
import cfg.expression.VariableRef;
import cfg.function.Function;
import cfg.variable.Variable;
import cfg.variable.VariableName;

public abstract class Matcher {
  protected IrItr stream;
  private String  error;

  public String getError() {
    return error;
  }

  public boolean hasError() {
    return error != null;
  }

  protected void wrongToken(IrType expected) {
    error = "Wrong token: " + stream.peek().getIrType() + ", expected: " + expected;
  }

  protected void doError(String msg, Object expected) {
    error = msg + ": " + stream.peek() + ", expeted: " + expected;
  }

  protected void checkForEnd() {
    if (!hasError()) {
      if (stream.hasNext()) {
        error = "Expected end of stream but found: " + stream.peek();
      }
    }
  }

  public void parse(IrItr stream) {
    this.stream = stream;
    error = null;
    doParse();
    this.stream = null;
  }

  abstract protected void doParse();

  protected boolean parseFuncRefLinked(Function func) {
    if (stream.peek().getIrType() == IrType.CallExprLinked) {
      if (((CallExprLinked) stream.peek()).getFunc() == func) {
        stream.next();
        return true;
      }
    }
    return false;
  }

  protected VariableName parseVarDef() {
    if (stream.peek().getIrType() == IrType.Variable) {
      return ((Variable) stream.next()).getName();
    } else {
      wrongToken(IrType.Variable);
    }
    return null;
  }

  protected VariableName parseVarRef() {
    switch (stream.peek().getIrType()) {
      case VariableRefKilled:
      case VariableRefLinked:
      case VariableRefUnlinked: {
        return ((VariableRef) stream.next()).getName();
      }
      default: {
        wrongToken(IrType.VariableRefUnlinked);
      }
    }
    return null;
  }

  protected long parseIntConstant() {
    if (stream.peek().getIrType() == IrType.IntConstant) {
      return ((IntConstant) stream.next()).getValue();
    } else {
      wrongToken(IrType.IntConstant);
    }
    return 0;
  }

  protected IntegerOp parseIntegerExpr() {
    if (stream.peek().getIrType() == IrType.IntegerExpr) {
      return ((IntegerExpr) stream.next()).getOp();
    } else {
      wrongToken(IrType.IntegerExpr);
    }
    return null;
  }

  protected int removeTokens(IrType token) {
    int num = 0;
    while (stream.peek().getIrType() == token) {
      stream.next();
      num++;
    }
    return num;
  }

  protected boolean removeToken(IrType token) {
    if (stream.peek().getIrType() != token) {
      wrongToken(token);
      return false;
    } else {
      stream.next();
      return true;
    }
  }

  protected void skipExpr() {
    switch (stream.peek().getIrType()) {
      case VariableRefKilled:
      case VariableRefLinked:
      case VariableRefUnlinked: {
        stream.next();
        break;
      }
      default: {
        throw new RuntimeException("Unknown expression: " + stream.peek().getIrType());
      }
    }
  }

}
