/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.matcher;

import reduction.ExprCopy;
import cfg.IntConstant;
import cfg.IrItr;
import cfg.IrType;
import cfg.expression.Expression;
import cfg.expression.IfExpr;
import cfg.expression.IntegerExpr;
import cfg.expression.IntegerOp;

// parses all pointer arithmetic expressions and converts them into an array index
public class PointerArithmeticMatcher extends Matcher {
  private Expression converted;

  public Expression getConverted() {
    return converted;
  }

  @Override
  protected void doParse() {
    converted = loparse();
    checkForEnd();
  }

  private Expression loparse() {
    switch (stream.peek().getIrType()) {
      case IntegerExpr: {
        return parseIntExpr();
      }
      case IntConstant: {
        return parseIntConstant4();
      }
      case IfExpr: {
        IfExpr expr = (IfExpr) stream.next();
        while (stream.hasNext()) {
          stream.next();
        }

        PointerArithmeticMatcher left = new PointerArithmeticMatcher();
        left.parse(new IrItr(expr.getLeft()));
        if (left.hasError()) {
          doError(left.getError(), expr);
          return null;
        }
        PointerArithmeticMatcher right = new PointerArithmeticMatcher();
        right.parse(new IrItr(expr.getRight()));
        if (right.hasError()) {
          doError(right.getError(), expr);
          return null;
        }

        return new IfExpr(ExprCopy.copy(expr.getCondition()), ExprCopy.copy(left.getConverted()), ExprCopy.copy(right
            .getConverted()));
      }
      default: {
        if (stream.peek() instanceof Expression) {
          return new IntegerExpr((Expression) stream.next(), new IntConstant(4), IntegerOp.Div);
        }
        wrongToken(IrType.IntegerExpr);
        return null;
      }
    }
  }

  private IntConstant parseIntConstant4() {
    IntConstant con = (IntConstant) stream.next();
    if (con.getValue() % 4 != 0) {
      doError("not a multiply of 4: " + con, null);
      return null;
    }
    return new IntConstant(con.getValue() / 4);
  }

  private IntConstant parseIntConstantShl() {
    IntConstant con = (IntConstant) stream.next();
    if (con.getValue() < 2) {
      doError("not a multiply of 4: " + con, null);
      return null;
    }
    return new IntConstant(1 << (con.getValue() - 2));
  }

  private Expression parseIntExpr() {
    IntegerExpr expr = (IntegerExpr) stream.next();
    switch (expr.getOp()) {
      case Mul: {
        if( stream.peek().getIrType() == IrType.IntConstant ){
          return new IntegerExpr(parseAll(), parseIntConstant4(), IntegerOp.Mul);
        } else {
          return new IntegerExpr(expr, new IntConstant(4), IntegerOp.Div);
        }
      }
      case ShiftLeft: {
        return new IntegerExpr(parseAll(), parseIntConstantShl(), IntegerOp.Mul);
      }
      case Add:
      case Sub: {
        return new IntegerExpr(loparse(), loparse(), expr.getOp());
      }
      default: {
        doError("not handled expression: " + expr, null);
        return null;
      }
    }
  }

  private Expression parseAll() {
    Expression expr = (Expression) stream.next();
    return expr;
  }

}
