/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.parser;


import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Queue;
import java.util.Set;

import cfg.Application;
import cfg.IrItr;
import cfg.basicblock.BasicBlock;
import cfg.function.PrgFunction;
import cfg.matcher.NoretCallMatcher;
import cfg.variable.VariableName;
import disassembler.diStorm3.DecomposedInst;
import disassembler.diStorm3.Registers;
import elfreader.ElfReader;

public class IntermediateFactory {
  private Queue<Long>                queue     = new LinkedList<Long>();
  private HashMap<Long, PrgFunction> functions = new HashMap<Long, PrgFunction>();
  private ArrayList<DecomposedInst>  instructions;
  private ElfReader                  elfReader;
  private long                       mainfunc;
  private Application                app;

  public IntermediateFactory(long mainfunc, ArrayList<DecomposedInst> instructions, ElfReader elfReader, Application app) {
    queue.add(mainfunc);
    this.app = app;
    this.instructions = instructions;
    this.elfReader = elfReader;
    this.mainfunc = mainfunc;
  }

  public void process() {
    while (!queue.isEmpty()) {
      long addr = queue.peek();
      PrgFunction func = parseFunction(addr);
      if (func != null) {
        functions.put(addr, func);
      }
      queue.remove(addr);
    }
    app.setFunctions(new ArrayList<PrgFunction>(functions.values()), functions.get(mainfunc));
  }

  private boolean isAddrInRange(long addr) {
    return (addr >= instructions.get(0).getAddress())
        && (addr <= instructions.get(instructions.size() - 1).getAddress());
  }

  private PrgFunction parseFunction(long entry) {
    if (isAddrInRange(entry)) {
      FunctionFactory func = new FunctionFactory(entry, entry == mainfunc, instructions, elfReader, app);
      func.process();
      PrgFunction res = new PrgFunction(entry, func.getBbs());
      PrologParser pparser = new PrologParser(res.getEntry().getCode());
      res.setPreserves(new LinkedList<VariableName>(pparser.getStack()));
      res.getPreserves().add(Registers.ESP); // TODO we assume the stack register is always preserved

      // FIXME parse pro/epilog correct (also for gzip and coreutils)
      for (BasicBlock bb : res.getExit()) {
        NoretCallMatcher noret = new NoretCallMatcher();
        noret.parse(new IrItr(bb.getCode().getLast()));
        if (noret.hasError() || noret.doesReturn()) {
          EpilogParser eparser = new EpilogParser(bb.getCode(), pparser.getStack(), pparser.getStacksize());
          if (!eparser.isSuccess()) {
            throw new RuntimeException("Error by parsing epilog: " + bb);
            // System.out.println( "Error by parsing epilog: " + NumPrint.toString(bb.getStartAddr()) );
          }
        }
      }

      res.setStacksize((int) pparser.getStacksize());
      res.setBackupSize(pparser.getStack().size() * 4);
      res.setUseEbpAsStackAddressing(pparser.useEbpAsStackAddressing());

      addFunctions(func.getFuncAddr());
      return res;
    } else {
      // assume it is a library function
      // throw new RuntimeException("Function not found: " + NumPrint.toString(entry));
      return null;
    }
  }

  private void addFunctions(Set<Long> funcAddr) {
    for (long addr : funcAddr) {
      if (!queue.contains(addr) && !functions.containsKey(addr)) {
        queue.add(addr);
      }
    }
  }

}
