/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.parser;


import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import cfg.Assignable;
import cfg.Flags;
import cfg.IntConstant;
import cfg.expression.IntegerOp;
import cfg.matcher.Matcher;
import cfg.matcher.MultiMatcher;
import cfg.matcher.PushMatcher;
import cfg.matcher.VarCopyMatcher;
import cfg.matcher.VarnameIntOpChangeMatcher;
import cfg.statement.AssignmentStmt;
import cfg.statement.PhiStmt;
import cfg.statement.Statement;
import cfg.variable.SsaVariable;
import cfg.variable.VariableName;
import disassembler.diStorm3.Registers;

/**
 * Parses and manipulates the prolog of an function. The variables pushed onto the stack and the stacksize for the local
 * variables is captures. The prolog is then marked as deleted.
 * 
 * @author urs
 * 
 */
public class PrologParser {
  private LinkedList<VariableName> stack                   = new LinkedList<VariableName>();
  private long                     stacksize               = 0;
  private boolean                  foundStacksize          = false;
  private boolean                  useEbpAsStackAddressing = false;

  public PrologParser(List<Statement> header) {
    super();
    parse(header);
  }

  public List<VariableName> getStack() {
    return stack;
  }

  public long getStacksize() {
    return stacksize;
  }

  public boolean useEbpAsStackAddressing() {
    return useEbpAsStackAddressing;
  }

  private void parse(List<Statement> header) {
    // System.out.println( NumPrint.toString(header.get(0).getOriginal().getAddress()) );

    Set<Matcher> matcher = new HashSet<Matcher>();
    matcher.add(new PushMatcher());
    matcher.add(new VarCopyMatcher(Registers.EBP, Registers.ESP));
    matcher.add(new VarnameIntOpChangeMatcher(Registers.ESP));
    MultiMatcher mm = new MultiMatcher(matcher);

    for (int i = 0; i < header.size(); i++) {
      Statement stmt = header.get(i);
      Set<? extends Matcher> res = mm.match(stmt);

      if (res.size() > 1) {
        throw new RuntimeException("Very unexpected behaviour");
      }
      if (res.isEmpty()) {
        break;
      }

      Matcher m = res.iterator().next();

      if (m instanceof PushMatcher) {
        stack.push(((PushMatcher) m).getVariable());
      } else if (m instanceof VarCopyMatcher) {
        useEbpAsStackAddressing = true;
        // stack.push(((VarCopyMatcher) m).getDst());
        header.add(0, new PhiStmt(new SsaVariable(((VarCopyMatcher) m).getDst(), 0)));
        i++;
      } else if (m instanceof VarnameIntOpChangeMatcher) {
        VarnameIntOpChangeMatcher viocm = (VarnameIntOpChangeMatcher) m;
        assert (!foundStacksize);
        assert (viocm.getOp() == IntegerOp.Sub);
        foundStacksize = true;
        stacksize = viocm.getValue();
      }
      stmt.setDeleted();
    }
    {
      header.add(0, new PhiStmt(new SsaVariable(Registers.ESP, 0)));
    }
    { // Direction flag is assumed to be 0, so we set it. It is also assumed that it is not given as parameter.
      LinkedList<Assignable> dst = new LinkedList<Assignable>();
      dst.add(new SsaVariable(Flags.Direction, 0));
      header.add(0, new AssignmentStmt(0, dst, new IntConstant(0)));
    }
  }
}
