/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.unlinker;


import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import cfg.Assignable;
import cfg.IrTraverser;
import cfg.IrType;
import cfg.basicblock.BasicBlock;
import cfg.expression.Expression;
import cfg.expression.VariableRefUnlinked;
import cfg.function.FuncProtParam;
import cfg.function.PrgFunction;
import cfg.statement.AssignmentStmt;
import cfg.statement.PhiStmt;
import cfg.statement.Statement;
import cfg.variable.SsaVariable;
import cfg.variable.VariableName;

/**
 * Simple but correct SSA destruction using Sreedhar method. Introduces a lot of additional variables.
 * 
 * Precondition: all variables have different names and version 0
 * 
 * @author urs
 * 
 */
public class SimpleSsaBacktranslation extends IrTraverser<Void, Void> {

  private PrgFunction func;
  private BasicBlock  phibb = null;

  public static void translate(Collection<PrgFunction> functions) {
    SimpleSsaBacktranslation translation = new SimpleSsaBacktranslation();
    translation.visit(functions, null);
  }

  @Override
  protected Void visitAssignmentStmt(AssignmentStmt obj, Void param) {
    super.visitAssignmentStmt(obj, param);
    if (obj.getSource().getIrType() == IrType.PhiStmt) {
      throw new RuntimeException("should not reach this code");
      // obj.setDeleted();
    }
    return null;
  }

  @Override
  protected Void visitPhiStmt(PhiStmt obj, Void param) {
    // a bit hacky code to not handle pseudo phi for arguments
    for (FuncProtParam fpam : func.getParam()) {
      if (fpam.name.equals(obj.getVarname().getName().toString())) {
        obj.setDeleted();
        return null;
      }
    }

    assert (func != null);
    VariableName tmpvar = new HllVariable(obj.getVarname().getName() + "_t");
    for (Long bbid : obj.getOption().keySet()) {
      BasicBlock bb = func.getGraph().getVertex(bbid);
      Expression expr = obj.getOption().get(bbid);

      SsaVariable var = new SsaVariable(tmpvar, 0);
      List<Assignable> dst = new ArrayList<Assignable>();
      dst.add(var);

      AssignmentStmt ass = new AssignmentStmt(0, dst, expr);

      bb.getCode().add(bb.getCode().size() - 1, ass);
    }

    SsaVariable var = new SsaVariable(obj.getVarname().getName(), 0);
    List<Assignable> dst = new ArrayList<Assignable>();
    dst.add(var);

    AssignmentStmt ass = new AssignmentStmt(0, dst, new VariableRefUnlinked(tmpvar));

    phibb.getCode().add(phibb.getCode().indexOf(obj), ass);
    obj.setDeleted();
    return null;
  }

  @Override
  protected Void visitPrgFunction(PrgFunction obj, Void param) {
    assert (func == null);
    func = obj;
    super.visitPrgFunction(obj, param);
    assert (func == obj);
    func = null;
    return null;
  }

  @Override
  protected Void visitBasicBlock(BasicBlock obj, Void param) {
    assert (phibb == null);
    phibb = obj;
    for (int i = 0; i < obj.getCode().size(); i++) {
      Statement itr = obj.getCode().get(i);
      visit(itr, param);
    }
    assert (phibb == obj);
    phibb = null;
    return null;
  }

}
