/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package elffile;

import java.io.IOException;

import util.NumPrint;

public class SectionHeader {

  public static final int SHT_NULL     = 0;
  public static final int SHT_PROGBITS = 1;
  public static final int SHT_SYMTAB   = 2;
  public static final int SHT_STRTAB   = 3;
  public static final int SHT_RELA     = 4;
  public static final int SHT_HASH     = 5;
  public static final int SHT_DYNAMIC  = 6;
  public static final int SHT_NOTE     = 7;
  public static final int SHT_NOBITS   = 8;
  public static final int SHT_REL      = 9;
  public static final int SHT_SHLIB    = 10;
  public static final int SHT_DYNSYM   = 11;
  public static final int SHT_LOPROC   = 0x70000000;
  public static final int SHT_HIPROC   = 0x7fffffff;
  public static final int SHT_LOUSER   = 0x80000000;
  public static final int SHT_HIUSER   = 0xffffffff;

  private String          name         = null;
  private int             nameNr;
  private int             type;
  private int             flags;
  private int             addr;
  private int             offset;
  private int             size;
  private int             link;
  private int             info;
  private int             addralign;
  private int             entsize;

  public SectionHeader(int nameNr, int type, int flags, int addr, int offset, int size, int link, int info,
      int addralign, int entsize) {
    super();
    this.nameNr = nameNr;
    this.type = type;
    this.flags = flags;
    this.addr = addr;
    this.offset = offset;
    this.size = size;
    this.link = link;
    this.info = info;
    this.addralign = addralign;
    this.entsize = entsize;
  }

  public static SectionHeader loadFromStream(ELFStreamReader esr) throws IOException {
    return new SectionHeader(esr.readE32Word(), esr.readE32Word(), esr.readE32Word(), esr.readE32Word(),
        esr.readE32Word(), esr.readE32Word(), esr.readE32Word(), esr.readE32Word(), esr.readE32Word(),
        esr.readE32Word());
  }

  public void dump() {
    if (name == null) {
      System.out.println("name    : " + NumPrint.toString(nameNr));
    } else {
      System.out.println("name    : " + name);
    }
    System.out.println("type    : " + NumPrint.toString(type));
    System.out.println("flags   : " + NumPrint.toString(flags));
    System.out.println("addr    : " + NumPrint.toString(addr));
    System.out.println("offset  : " + NumPrint.toString(offset));
    System.out.println("size    : " + NumPrint.toString(size));
    System.out.println("link    : " + NumPrint.toString(link));
    System.out.println("info    : " + NumPrint.toString(info));
    System.out.println("align   : " + NumPrint.toString(addralign));
    System.out.println("entsize : " + NumPrint.toString(entsize));
    System.out.println();
  }

  public int getFlags() {
    return flags;
  }

  public int getType() {
    return type;
  }

  public int getOffset() {
    return offset;
  }

  public int getNameNr() {
    return nameNr;
  }

  public String getName() {
    return name;
  }

  public void setName(String name) {
    this.name = name;
  }

  public int size() {
    return size;
  }

  public int addr() {
    return addr;
  }

  public int offset() {
    return offset;
  }

  public boolean isInSection(long offset) {
    return (offset >= addr()) && (offset < addr() + size());
  }

  public long toFileOffset(long offset) {
    return offset + this.offset - this.addr;
  }

}
