/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package knowledge;


import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import cfg.Flags;
import cfg.IrTraverser;
import cfg.function.Function;
import cfg.function.LibFunction;
import cfg.function.PrgFunction;
import cfg.variable.SsaVariable;
import cfg.variable.VariableName;
import disassembler.diStorm3.Registers;

/**
 * Variables killed by the function, excluding called functions
 * 
 * @author urs
 * 
 */
public class KnowLocalKilled extends KnowledgeEntry {
  private HashMap<Function, Set<VariableName>> killed = new HashMap<Function, Set<VariableName>>();
  private KnowledgeBase                      base;

  @Override
  public void init(KnowledgeBase base) {
    this.base = base;
  }

  private void rebuild() {
    killed = new HashMap<Function, Set<VariableName>>();
    KnowLocalKilledVisitor visitor = new KnowLocalKilledVisitor(killed);
    visitor.visit(base.getApp().getFunctions(), null);
    /*
     * for (Function f : SystemFunctions.getAll()) { killed.put(f, new HashSet<VariableName>()); }
     */
  }

  public Set<VariableName> getKills(Function func) {
    switch (func.getIrType()) {
      case FuncLibrary: {
        return libKilled();
      }
      case FuncProgram: {
        if (!killed.containsKey(func)) {
          rebuild();
        }
        if (!killed.containsKey(func)) {
          throw new RuntimeException("Can not find function: " + func);
        }
        return killed.get(func);
      }
      case FuncSystem: {
        return new HashSet<VariableName>();
      }
      default: {
        throw new RuntimeException("Not yet handled function type: " + func.getIrType());
      }
    }
  }

  private Set<VariableName> libKilled() {
    HashSet<VariableName> cdeclKill = new HashSet<VariableName>();
    for (Flags f : Flags.values()) {
      cdeclKill.add(f);
    }
    cdeclKill.add(Registers.EAX);
    cdeclKill.add(Registers.ECX);
    cdeclKill.add(Registers.EDX);
    cdeclKill.add(Registers.ST0);
    cdeclKill.add(Registers.ST1);
    cdeclKill.add(Registers.ST2);
    cdeclKill.add(Registers.ST3);
    cdeclKill.add(Registers.ST4);
    cdeclKill.add(Registers.ST5);
    cdeclKill.add(Registers.ST6);
    cdeclKill.add(Registers.ST7);
    cdeclKill.add(Registers.XMM0);
    cdeclKill.add(Registers.XMM1);
    cdeclKill.add(Registers.XMM2);
    cdeclKill.add(Registers.XMM3);
    cdeclKill.add(Registers.XMM4);
    cdeclKill.add(Registers.XMM5);
    cdeclKill.add(Registers.XMM6);
    cdeclKill.add(Registers.XMM7);
    cdeclKill.add(Registers.YMM0);
    cdeclKill.add(Registers.YMM1);
    cdeclKill.add(Registers.YMM2);
    cdeclKill.add(Registers.YMM3);
    cdeclKill.add(Registers.YMM4);
    cdeclKill.add(Registers.YMM5);
    cdeclKill.add(Registers.YMM6);
    cdeclKill.add(Registers.YMM7);
    return cdeclKill;
  }

}

class KnowLocalKilledVisitor extends IrTraverser<Void, Function> {
  private HashMap<Function, Set<VariableName>> kills;

  public KnowLocalKilledVisitor(HashMap<Function, Set<VariableName>> killed) {
    kills = killed;
  }

  @Override
  protected Void visitSsaVariable(SsaVariable obj, Function param) {
    if (!obj.getName().onlyLocalScope()) {
      kills.get(param).add(obj.getName());
    }
    return null;
  }

  @Override
  protected Void visitPrgFunction(PrgFunction obj, Function param) {
    assert (param == null);
    kills.put(obj, new HashSet<VariableName>());
    return super.visitPrgFunction(obj, obj);
  }

  @Override
  protected Void visitLibFunction(LibFunction obj, Function param) {
    assert (param == null);
    throw new RuntimeException("should not reached this point");
    // because library functions should not be added to the program
    // return null;
  }

}
