/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package knowledge;


import java.util.LinkedList;
import java.util.List;

import cfg.IntConstant;
import cfg.IrItr;
import cfg.IrTraverser;
import cfg.expression.CallExpr;
import cfg.expression.CallExprLinked;
import cfg.expression.CallExprPointer;
import cfg.expression.CallExprUnlinked;
import cfg.matcher.VarAndOffsetMatcher;
import cfg.statement.AssignmentStmt;
import cfg.statement.Statement;
import cfg.variable.ArrayAccess;
import cfg.variable.GlobalVariable;
import cfg.variable.StackVariable;
import cfg.variable.VariablePtrDeref;
import cfg.variable.VariablePtrOf;

public class KnowMemAccess extends KnowledgeEntry {
  private KnowMemAccessVisitor cache = null;

  private void updateCache(Statement stmt) {
    if ((cache != null) && (cache.getStmt() == stmt)) {
      return;
    }
    cache = new KnowMemAccessVisitor();
    cache.visit(stmt, null);
  }

  @Override
  public void init(KnowledgeBase base) {
  }

  public boolean doesDynamicRead(Statement stmt) {
    updateCache(stmt);
    return cache.getDynamicRead();
  }

  public boolean doesDynamicWrite(Statement stmt) {
    updateCache(stmt);
    return cache.getDynamicWrite();
  }

  public boolean doesRead(Statement stmt) {
    return doesDynamicRead(stmt) || !getStaticReads(stmt).isEmpty();
  }

  public boolean doesWrite(Statement stmt) {
    return doesDynamicWrite(stmt) || !getStaticWrites(stmt).isEmpty();
  }

  public List<MemAccess> getStaticReads(Statement stmt) {
    updateCache(stmt);
    return cache.getStaticReads();
  }

  public List<MemAccess> getStaticWrites(Statement stmt) {
    updateCache(stmt);
    return cache.getStaticWrites();
  }
}

// param is false if we are on the rhs (source) side of an assignment it is true on the lhs (destination)
class KnowMemAccessVisitor extends IrTraverser<Void, Boolean> {
  private boolean         dynamicRead  = false;
  private boolean         dynamicWrite = false;
  private List<MemAccess> staticReads  = new LinkedList<MemAccess>();
  private List<MemAccess> staticWrites = new LinkedList<MemAccess>();
  private Statement       stmt         = null;                       // actual owner statement

  public Statement getStmt() {
    return stmt;
  }

  public List<MemAccess> getStaticReads() {
    return staticReads;
  }

  public List<MemAccess> getStaticWrites() {
    return staticWrites;
  }

  public boolean getDynamicRead() {
    return dynamicRead;
  }

  public boolean getDynamicWrite() {
    return dynamicWrite;
  }

  private void accessDynamicMemory(Boolean lhs) {
    assert (lhs != null);
    if (lhs) {
      dynamicWrite = true;
    } else {
      dynamicRead = true;
    }
  }

  @Override
  protected Void visitVariablePtrOf(VariablePtrOf obj, Boolean param) {
    assert (!param);
    switch (obj.getExpression().getIrType()) {
      case VariableArray: {
        return visit(((ArrayAccess) obj.getExpression()).getIndex(), param);
      }
      default: {
        return visit(obj.getExpression(), param);
      }
    }
  }

  @Override
  protected Void visitVariableArrayAccess(ArrayAccess obj, Boolean param) {
    if (obj.getIndex() instanceof IntConstant) {
      accessStaticMemory(obj.getBase(), ((IntConstant) obj.getIndex()).getValue() * 4, param); // FIXME *4 depends on
                                                                                               // the element type size
    } else {
      accessDynamicMemory(param);
    }
    return super.visitVariableArrayAccess(obj, false);
  }

  @Override
  protected Void visitVariablePtrDeref(VariablePtrDeref obj, Boolean param) {
    VarAndOffsetMatcher matcher = new VarAndOffsetMatcher();
    matcher.parse(new IrItr(obj.getExpression()));
    if (matcher.hasError()) {
      accessDynamicMemory(param);
    } else {
      accessStaticMemory(matcher.getVar(), matcher.getOffset(), param);
    }
    return super.visitVariablePtrDeref(obj, false);
  }

  private void accessStaticMemory(Object var, long offset, Boolean rhs) {
    MemAccess mem = new MemAccess();
    mem.base = var;
    mem.offset = offset;
    mem.size = 4; // FIXME determine correct value
    assert (rhs != null);
    if (rhs) {
      staticWrites.add(mem);
    } else {
      staticReads.add(mem);
    }
  }

  @Override
  protected Void visitStackVariable(StackVariable obj, Boolean param) {
    assert (false);
    accessDynamicMemory(param);
    return super.visitStackVariable(obj, param);
  }

  @Override
  protected Void visitGlobalVariable(GlobalVariable obj, Boolean param) {
    accessStaticMemory(obj.getSegment(), obj.getAddress(), param);
    return super.visitGlobalVariable(obj, param);
  }

  @Override
  protected Void visitCallExprLinked(CallExprLinked obj, Boolean param) {
    return visitCall(obj);
  }

  @Override
  protected Void visitCallExprUnlinked(CallExprUnlinked obj, Boolean param) {
    return visitCall(obj);
  }

  @Override
  protected Void visitCallExprPointer(CallExprPointer obj, Boolean param) {
    return visitCall(obj);
  }

  private Void visitCall(CallExpr obj) {
    // TODO too pessimistic
    dynamicWrite = true;
    dynamicRead = true;
    return null;
  }

  @Override
  protected Void visitStatement(Statement obj, Boolean param) {
    assert (param == null);
    assert (stmt == null);
    stmt = obj;
    super.visitStatement(obj, false);
    return null;
  }

  @Override
  protected Void visitAssignmentStmt(AssignmentStmt obj, Boolean param) {
    assert (param == false);
    visit(obj.getSource(), false);
    visit(obj.getDestination(), true);
    return null;
  }
}
