/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package knowledge;


import java.util.HashMap;

import cfg.IrTraverser;
import cfg.basicblock.BasicBlock;
import cfg.expression.Expression;
import cfg.function.PrgFunction;
import cfg.statement.Statement;
import cfg.variable.SsaVariable;
import cfg.variable.Variable;

public class KnowOwner extends KnowledgeEntry {
  private HashMap<Variable, Statement>     varOwner  = new HashMap<Variable, Statement>();
  private HashMap<Expression, Statement>   exprOwner = new HashMap<Expression, Statement>();
  private HashMap<Statement, BasicBlock>   stmtOwner = new HashMap<Statement, BasicBlock>();
  private HashMap<BasicBlock, PrgFunction> bbOwner   = new HashMap<BasicBlock, PrgFunction>();
  private KnowledgeBase                  base;

  @Override
  public void init(KnowledgeBase base) {
    this.base = base;
  }

  private void rebuild() {
    varOwner = new HashMap<Variable, Statement>();
    exprOwner = new HashMap<Expression, Statement>();
    stmtOwner = new HashMap<Statement, BasicBlock>();
    bbOwner = new HashMap<BasicBlock, PrgFunction>();
    KnowOwnerVisitor visitor = new KnowOwnerVisitor(varOwner, exprOwner, stmtOwner, bbOwner);
    visitor.visit(base.getApp().getFunctions(), null);
  }

  public Statement getVarOwner(Variable var) {
    if (!varOwner.containsKey(var)) {
      rebuild();
    }
    if (!varOwner.containsKey(var)) {
      throw new RuntimeException("Can not find variable owner: " + var);
    }
    return varOwner.get(var);
  }

  public Statement getExprOwner(Expression expr) {
    if (!exprOwner.containsKey(expr)) {
      rebuild();
    }
    if (!exprOwner.containsKey(expr)) {
      throw new RuntimeException("Can not find expression owner: " + expr);
    }
    return exprOwner.get(expr);
  }

  public BasicBlock getStmtOwner(Statement stmt) {
    if (!stmtOwner.containsKey(stmt)) {
      rebuild();
    }
    if (!stmtOwner.containsKey(stmt)) {
      throw new RuntimeException("Can not find statement owner: " + stmt);
    }
    return stmtOwner.get(stmt);
  }

  public PrgFunction getBbOwner(BasicBlock bb) {
    if (!bbOwner.containsKey(bb)) {
      rebuild();
    }
    if (!bbOwner.containsKey(bb)) {
      throw new RuntimeException("Can not find basic block owner: " + bb);
    }
    return bbOwner.get(bb);
  }

}

class KnowOwnerVisitor extends IrTraverser<Void, Void> {
  private HashMap<Variable, Statement>     varOwner;
  private HashMap<Expression, Statement>   exprOwner;
  private HashMap<Statement, BasicBlock>   stmtOwner;
  private HashMap<BasicBlock, PrgFunction> bbOwner;

  private PrgFunction                      func;
  private BasicBlock                       bb;
  private Statement                        stmt;

  public KnowOwnerVisitor(HashMap<Variable, Statement> varOwner, HashMap<Expression, Statement> exprOwner,
      HashMap<Statement, BasicBlock> stmtOwner, HashMap<BasicBlock, PrgFunction> bbOwner) {
    super();
    this.varOwner = varOwner;
    this.exprOwner = exprOwner;
    this.stmtOwner = stmtOwner;
    this.bbOwner = bbOwner;
  }

  @Override
  protected Void visitPrgFunction(PrgFunction obj, Void param) {
    assert (func == null);
    func = obj;
    super.visitPrgFunction(obj, null);
    func = null;
    return null;
  }

  @Override
  protected Void visitBasicBlock(BasicBlock obj, Void param) {
    assert (bb == null);
    bb = obj;
    bbOwner.put(bb, func);
    super.visitBasicBlock(obj, null);
    bb = null;
    return null;
  }

  @Override
  protected Void visitStatement(Statement obj, Void param) {
    if (!obj.isDeleted()) {
      assert (stmt == null);
      stmt = obj;
      stmtOwner.put(stmt, bb);
      super.visitStatement(obj, null);
      stmt = null;
    }
    return null;
  }

  @Override
  protected Void visitExpression(Expression obj, Void param) {
    exprOwner.put(obj, stmt);
    return super.visitExpression(obj, param);
  }

  @Override
  protected Void visitSsaVariable(SsaVariable obj, Void param) {
    varOwner.put(obj, stmt);
    return super.visitSsaVariable(obj, param);
  }

}
