/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package knowledge;


import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import cfg.IrTraverser;
import cfg.expression.VariableRefLinked;
import cfg.statement.Statement;
import cfg.variable.Array;
import cfg.variable.SsaVariable;
import cfg.variable.Variable;

/**
 * Knows the referencees of all definitions (i.e. which statements depends on the one specified)
 * 
 * @author urs
 * 
 */
public class KnowReferencees extends KnowledgeEntry {
  private Map<Variable, Set<Statement>> map = new HashMap<Variable, Set<Statement>>();
  private KnowledgeBase               base;

  @Override
  public void init(KnowledgeBase base) {
    this.base = base;
  }

  private void rebuild() {
    map = new HashMap<Variable, Set<Statement>>();
    KnowOwner owner = (KnowOwner) base.getEntry(KnowOwner.class);
    KnowReferenceesVisitor visitor = new KnowReferenceesVisitor(map, owner);
    visitor.visit(base.getApp().getFunctions(), null);
  }

  public Set<Statement> getReferencees(Variable var) {
    if (!map.containsKey(var)) {
      rebuild();
    }
    if (!map.containsKey(var)) {
      throw new RuntimeException("Can not find definition of variable: " + var);
    }
    return map.get(var);
  }

}

class KnowReferenceesVisitor extends IrTraverser<Void, Void> {
  private Map<Variable, Set<Statement>> map;
  private KnowOwner                     owner;

  public KnowReferenceesVisitor(Map<Variable, Set<Statement>> map, KnowOwner owner) {
    super();
    this.owner = owner;
    this.map = map;
  }

  @Override
  protected Void visitSsaVariable(SsaVariable obj, Void param) {
    addDef(obj);
    return null;
  }

  @Override
  protected Void visitArray(Array obj, Void param) {
    addDef(obj);
    return null;
  }

  @Override
  protected Void visitStatement(Statement obj, Void param) {
    assert (param == null);
    super.visitStatement(obj, null);
    return null;
  }

  @Override
  protected Void visitVariableRefLinked(VariableRefLinked obj, Void param) {
    Variable key = obj.getReference();
    Statement val = owner.getExprOwner(obj);
    addDef(key);
    map.get(key).add(val);
    return super.visitVariableRefLinked(obj, param);
  }

  private void addDef(Variable key) {
    if (!map.containsKey(key)) {
      map.put(key, new HashSet<Statement>());
    }
  }
}
