/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package knowledge.memory;

import cfg.IntConstant;
import cfg.IrItr;
import cfg.IrTraverser;
import cfg.expression.CallExpr;
import cfg.expression.CallExprLinked;
import cfg.expression.CallExprPointer;
import cfg.expression.CallExprUnlinked;
import cfg.function.Function;
import cfg.function.system.SystemFunctions;
import cfg.matcher.VarAndOffsetMatcher;
import cfg.statement.AssignmentStmt;
import cfg.statement.PhiStmt;
import cfg.statement.Statement;
import cfg.variable.ArrayAccess;
import cfg.variable.GlobalVariable;
import cfg.variable.StackVariable;
import cfg.variable.VariablePtrDeref;

public class MemoryWriteTraverser extends IrTraverser<Void, Boolean> {
  protected Statement         stmt          = null;                                                          // actual
  protected Memory<Statement> memory        = new Memory<Statement>();
  private final Function[]    funcWhitelist = { SystemFunctions.castInt64, SystemFunctions.halt,
      SystemFunctions.readInt, SystemFunctions.writeInt, SystemFunctions.writeNl, SystemFunctions.writeStr };

  public Memory<Statement> getMemory() {
    return memory;
  }

  protected void accessDynamicMemory(boolean write) {
    if (write) {
      memory.setDynamicWriter(stmt);
    }
  }

  protected void accessStaticMemory(int offset, boolean write) {
    if (write) {
      memory.setWriter(offset, stmt);
    }
  }

  @Override
  protected Void visitVariableArrayAccess(ArrayAccess obj, Boolean param) {
    if (obj.getIndex() instanceof IntConstant) {
      accessStaticMemory((int) ((IntConstant) obj.getIndex()).getValue(), param);
    } else {
      accessDynamicMemory(param);
    }
    return super.visitVariableArrayAccess(obj, false);
  }

  @Override
  protected Void visitVariablePtrDeref(VariablePtrDeref obj, Boolean param) {
    VarAndOffsetMatcher matcher = new VarAndOffsetMatcher();
    matcher.parse(new IrItr(obj.getExpression()));
    if (matcher.hasError()) {
      accessDynamicMemory(param);
    } else {
      assert (matcher.getOffset() % 4 == 0);
      accessStaticMemory((int) (matcher.getOffset() / 4), param);
    }
    return super.visitVariablePtrDeref(obj, false);
  }

  @Override
  protected Void visitStackVariable(StackVariable obj, Boolean param) {
    assert (false);
    accessDynamicMemory(param);
    return super.visitStackVariable(obj, param);
  }

  @Override
  protected Void visitGlobalVariable(GlobalVariable obj, Boolean param) {
    assert (false);
    accessStaticMemory((int) obj.getAddress(), param);
    return super.visitGlobalVariable(obj, param);
  }

  @Override
  protected Void visitCallExprLinked(CallExprLinked obj, Boolean param) {
    super.visitCallExprLinked(obj, param);
    if (!isInWhitelist(obj.getFunc())) {
      visitCall(obj);
    }
    return null;
  }

  private boolean isInWhitelist(Function func) {
    for (Function white : funcWhitelist) {
      if (white == func) {
        return true;
      }
    }
    return false;
  }

  @Override
  protected Void visitCallExprUnlinked(CallExprUnlinked obj, Boolean param) {
    super.visitCallExprUnlinked(obj, param);
    visitCall(obj);
    return null;
  }

  @Override
  protected Void visitCallExprPointer(CallExprPointer obj, Boolean param) {
    super.visitCallExprPointer(obj, param);
    visitCall(obj);
    return null;
  }

  private void visitCall(CallExpr obj) {
    // FIXME too pessimistic
    accessDynamicMemory(true);
  }

  @Override
  protected Void visitStatement(Statement obj, Boolean param) {
    assert (param == null);
    assert (stmt == null);
    stmt = obj;
    super.visitStatement(obj, false);
    assert (stmt == obj);
    stmt = null;
    return null;
  }

  @Override
  protected Void visitAssignmentStmt(AssignmentStmt obj, Boolean param) {
    assert (param == false);
    visit(obj.getSource(), false);
    visit(obj.getDestination(), true);
    return null;
  }

  @Override
  protected Void visitPhiStmt(PhiStmt obj, Boolean param) {
    assert (param == false);
    visit(obj.getOption().values(), false);
    visit(obj.getDestination(), true);
    return null;
  }
}
