/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package phases.ast;

import graph.SimpleEdge;
import graph.SimpleUndirected;
import graph.TestGraph;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jgrapht.alg.ChromaticNumber;

import phases.ArtefactType;
import phases.Phase;
import phases.Upcompiler;
import util.Pair;
import ast.PrgFunc;
import ast.VariableMerger;
import ast.traverser.AstTraverser;
import ast.traverser.AstVarDependency;
import ast.traverser.AstVariableCollector;
import ast.traverser.NopAstStatementTraverser;
import ast.traverser.VariableAccessCollector;
import ast.type.Type;
import ast.variable.Variable;
import cfg.Application;
import dataflow.DfVertex;
import dataflow.livevar.LiveVarGb;
import dataflow.livevar.LiveVariableAlgo;

public class Coalescing extends Phase {
  private int removed;

  public Coalescing(Upcompiler upcompiler) {
    super(upcompiler);
  }

  public final int getRemoved() {
    return removed;
  }

  @Override
  public String getDescription() {
    return "coalescing variables";
  }

  @Override
  public boolean process(Application app) {
    removed = 0;
    for (PrgFunc func : app.getPrg().getFunction()) {

      List<Variable> vars = AstVariableCollector.process(func);

      // create graph for coloring
      SimpleUndirected<Variable> graph = new SimpleUndirected<Variable>();
      for (Variable var : vars) {
        graph.addVertex(var);
      }

      addTypeRestrictions(func, graph);
      addLiverangeRestrictions(func, graph);
      addUnrelatedRestrictions(func, vars, graph);

      // DotSimpleWriter<Variable, Pair<Variable, Variable>> writer2 = new DotSimpleWriter<Variable, Pair<Variable,
      // Variable>>(
      // new VerySimpleLabel<Variable, Pair<Variable, Variable>>(), graph.complement());
      // writer2.write(filename + "." + func.getName() + ".color.gv");

      Map<Integer, Set<Variable>> colors = ChromaticNumber.findGreedyColoredGroups(graph);
      Set<List<Variable>> color = new HashSet<List<Variable>>();
      for (Set<Variable> var : colors.values()) {
        ArrayList<Variable> list = new ArrayList<Variable>(var);
        Collections.sort(list);
        color.add(list);
      }
      removed += VariableMerger.merge(func, color);

    }
    return true;
  }

  private static void addUnrelatedRestrictions(PrgFunc func, List<Variable> vars, SimpleUndirected<Variable> graph) {
    // adds restrictions for unrelated variables
    // transitive closure
    // undirected complement
    TestGraph<Variable> depgraph = AstVarDependency.process(func);
    transitiveClosure(depgraph);

    SimpleUndirected<Variable> udep = new SimpleUndirected<Variable>();
    for (Variable var : vars) {
      udep.addVertex(var);
    }
    for (Variable u : vars) {
      for (Variable v : vars) {
        if (depgraph.containsEdge(u, v)) {
          udep.addEdge(u, v);
        }
      }
    }
    udep = udep.complement();
    for (Pair<Variable, Variable> e : udep.edgeSet()) {
      if (!e.getFirst().equals(e.getSecond())) {
        graph.addEdge(e.getFirst(), e.getSecond());
      }
    }
  }

  private static void addLiverangeRestrictions(PrgFunc func, SimpleUndirected<Variable> graph) {
    Set<List<Variable>> loclive = getLiveRange(func);
    for (List<Variable> samelive : loclive) {
      // samelive = new HashSet<Variable>(samelive);
      for (Variable x : samelive) {
        for (Variable y : samelive) {
          if (x != y) {
            graph.addEdge(x, y);
          }
        }
      }
    }
  }

  private static void addTypeRestrictions(PrgFunc func, SimpleUndirected<Variable> graph) {
    Map<Type, Set<Variable>> typed = getTypeBuckets(func);
    for (Type typA : typed.keySet()) {
      for (Type typB : typed.keySet()) {
        if (!typA.equals(typB)) {
          for (Variable varA : typed.get(typA)) {
            for (Variable varB : typed.get(typB)) {
              graph.addEdge(varA, varB);
            }
          }
        }
      }
    }
  }

  private static void transitiveClosure(TestGraph<Variable> depgraph) {
    // more or less FloydWarshall
    for (Variable k : depgraph.vertexSet()) {
      for (Variable i : depgraph.vertexSet()) {
        for (Variable j : depgraph.vertexSet()) {
          if (depgraph.containsEdge(i, k) && depgraph.containsEdge(k, j)) {
            depgraph.addEdge(i, j);
          }
        }
      }
    }
  }

  private static Map<Type, Set<Variable>> getTypeBuckets(PrgFunc func) {
    Map<Type, Set<Variable>> typed = new HashMap<Type, Set<Variable>>();

    { // get bucket of variables with same type
      VariableAccessCollector collector = new VariableAccessCollector();
      AstTraverser<Void> traverser = new AstTraverser<Void>(new NopAstStatementTraverser<Void>(collector));
      traverser.visit(func, null);
      Set<Variable> vars = new HashSet<Variable>(collector.getDefined());
      // vars.removeAll(func.getParam()); // remove parameter
      for (Variable var : vars) {
        if (var.getSize() > 1) {
          continue;
        }
        Set<Variable> typ;
        if (typed.containsKey(var.getType())) {
          typ = typed.get(var.getType());
        } else {
          typ = new HashSet<Variable>();
          typed.put(var.getType(), typ);
        }
        typ.add(var);
      }
    }
    return typed;
  }

  private static Set<List<Variable>> getLiveRange(PrgFunc func) {
    LiveVarGb builder = new LiveVarGb();
    builder.build(func);
    LiveVariableAlgo<Variable, SimpleEdge<DfVertex<Variable>>> algo = new LiveVariableAlgo<Variable, SimpleEdge<DfVertex<Variable>>>(
        builder.getGraph());
    algo.process();

    Set<List<Variable>> loclive = new HashSet<List<Variable>>();
    Map<ast.statement.Statement, List<Variable>> live = new HashMap<ast.statement.Statement, List<Variable>>();
    Map<ast.statement.Statement, DfVertex<Variable>> mapping = builder.getMapping();
    for (ast.statement.Statement stmt : mapping.keySet()) {
      List<Variable> lst = new ArrayList<Variable>(algo.getOut().get(mapping.get(stmt)));
      Collections.sort(lst);
      live.put(stmt, lst);
      loclive.add(lst);
    }

    // LiveWriter<DfVertex<Variable>> writer = new LiveWriter<DfVertex<Variable>>(builder.getGraph(),
    // builder.getInvMapping());
    // writer.write(filename + "." + func.getName() + ".gv");
    return loclive;
  }

  @Override
  public ArtefactType outArtefact() {
    return ArtefactType.AST;
  }

}
