/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package reduction;

import cfg.Condition;
import cfg.IntConstant;
import cfg.expression.BooleanExpr;
import cfg.expression.CompareExpr;
import cfg.expression.CompareOp;
import cfg.expression.Expression;
import cfg.expression.IntegerExpr;
import cfg.expression.VariableRefLinked;
import cfg.statement.AssignmentStmt;
import cfg.statement.Statement;
import cfg.variable.Variable;

public class ConditionParser {

  public static BooleanExpr findExpression(Condition cond, Statement cmp) {
    switch (cmp.getOriginal().getOpcode()) {
      case TEST:
      case AND:
        return findExprTest(cond, (AssignmentStmt) cmp);
      case CMP:
        return findExprCmp(cond, (AssignmentStmt) cmp);
      case DEC:
      case ADD:
        return findExprDec(cond, (AssignmentStmt) cmp);
      default:
        throw new RuntimeException("Not yet handled opcode: " + cmp.getOriginal().getOpcode());
    }
  }

  private static BooleanExpr findExprDec(Condition cond, AssignmentStmt cmp) {
    Variable expr = (Variable) cmp.getDestination().iterator().next();

    VariableRefLinked ref = new VariableRefLinked(expr);

    switch (cond) {
      case NotEqualNotZero:
        return new CompareExpr(ref, new IntConstant(0), CompareOp.NOT_EQUAL);
      case EqualZero:
        return new CompareExpr(ref, new IntConstant(0), CompareOp.EQUAL);
      default:
        throw new RuntimeException("Not yet handled opcode: " + cond);
    }
  }

  private static BooleanExpr findExprCmp(Condition cond, AssignmentStmt cmp) {
    IntegerExpr expr = (IntegerExpr) cmp.getSource();

    switch (cond) {
      case EqualZero:
        return new CompareExpr(ExprCopy.copy(expr.getLeft()), ExprCopy.copy(expr.getRight()), CompareOp.EQUAL);
      case LessThanOrEqual:
        return new CompareExpr(ExprCopy.copy(expr.getLeft()), ExprCopy.copy(expr.getRight()), CompareOp.LOWER_EQUAL);
      case BelowOrEqual:
        return new CompareExpr(ExprCopy.copy(expr.getLeft()), ExprCopy.copy(expr.getRight()), CompareOp.LOWER_EQUAL);
      case LessThan:
        return new CompareExpr(ExprCopy.copy(expr.getLeft()), ExprCopy.copy(expr.getRight()), CompareOp.LOWER);
      case Below:
        return new CompareExpr(ExprCopy.copy(expr.getLeft()), ExprCopy.copy(expr.getRight()), CompareOp.LOWER);
      case GreaterThanOrEqual:
        return new CompareExpr(ExprCopy.copy(expr.getLeft()), ExprCopy.copy(expr.getRight()), CompareOp.GREATER_EQUAL);
      case GreaterThan:
        return new CompareExpr(ExprCopy.copy(expr.getLeft()), ExprCopy.copy(expr.getRight()), CompareOp.GREATER);
      case Above:
        return new CompareExpr(ExprCopy.copy(expr.getLeft()), ExprCopy.copy(expr.getRight()), CompareOp.GREATER);
      case AboveOrEqual:
        return new CompareExpr(ExprCopy.copy(expr.getLeft()), ExprCopy.copy(expr.getRight()), CompareOp.GREATER_EQUAL);
      case NotEqualNotZero:
        return new CompareExpr(ExprCopy.copy(expr.getLeft()), ExprCopy.copy(expr.getRight()), CompareOp.NOT_EQUAL);
      default:
        throw new RuntimeException("Not yet handled opcode: " + cond);
    }
  }

  private static BooleanExpr findExprTest(Condition cond, AssignmentStmt cmp) {
    IntegerExpr expr = (IntegerExpr) cmp.getSource();
    Expression left = expr.getLeft();
    Expression right = expr.getRight();

    // compare with itself?
    if ((right instanceof VariableRefLinked) && (left instanceof VariableRefLinked)
        && (((VariableRefLinked) left).getReference() == ((VariableRefLinked) right).getReference())) {
      Variable var = ((VariableRefLinked) left).getReference();
      VariableRefLinked ref = new VariableRefLinked(var);

      switch (cond) {
        case LessThanOrEqual:
          return new CompareExpr(ref, new IntConstant(0), CompareOp.LOWER_EQUAL);
        case NotEqualNotZero:
          return new CompareExpr(ref, new IntConstant(0), CompareOp.NOT_EQUAL);
        case EqualZero:
          return new CompareExpr(ref, new IntConstant(0), CompareOp.EQUAL);
        case GreaterThan:
          return new CompareExpr(ref, new IntConstant(0), CompareOp.GREATER);
        case Sign:
          return new CompareExpr(ref, new IntConstant(0), CompareOp.LOWER); // TODO: check
        default:
          throw new RuntimeException("Not yet handled opcode: " + cond);
      }
    }

    switch (cond) {
      case EqualZero:
        return new CompareExpr(ExprCopy.copy(expr), new IntConstant(0), CompareOp.EQUAL);
      case NotEqualNotZero:
        return new CompareExpr(ExprCopy.copy(expr), new IntConstant(0), CompareOp.NOT_EQUAL);
      default:
        throw new RuntimeException("Not yet handled opcode: " + cond);
    }
  }

}
