/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package reduction;


import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;

import knowledge.KnowOwner;
import knowledge.KnowReferencees;
import knowledge.KnowledgeBase;
import cfg.Assignable;
import cfg.IrElement;
import cfg.IrReplaceExprTraverser;
import cfg.IrType;
import cfg.basicblock.BasicBlock;
import cfg.expression.Expression;
import cfg.expression.VariableRefLinked;
import cfg.statement.Assignment;
import cfg.statement.AssignmentStmt;
import cfg.statement.PhiStmt;
import cfg.statement.Statement;
import cfg.variable.Variable;

public class ExpressionReduction extends IrReplaceExprTraverser<Void> {
  private MemoryModelEnforcer mm;
  private BasicBlock          bb   = null;
  private Statement           stmt = null;
  private KnowledgeBase     kb;

  static public void reduce(IrElement obj, MemoryModelEnforcer mm, KnowledgeBase kb) {
    ExpressionReduction red = new ExpressionReduction(mm, kb);
    red.visit(obj, null);
  }

  public ExpressionReduction(MemoryModelEnforcer mm, KnowledgeBase kb) {
    super();
    this.mm = mm;
    this.kb = kb;
  }

  @Override
  protected void visitBasicBlock(BasicBlock obj, Void param) {
    assert (bb == null);
    bb = obj;
    super.visitBasicBlock(obj, param);
    bb = null;
  }

  @Override
  protected void visitStatement(Statement obj, Void param) {
    assert (stmt == null);
    stmt = obj;
    super.visitStatement(obj, param);
    stmt = null;
  }

  @Override
  protected Expression visitVariableRefLinked(VariableRefLinked obj, Void param) {
    KnowOwner owner = (KnowOwner) kb.getEntry(KnowOwner.class);
    Statement own = owner.getVarOwner(obj.getReference());

    assert (!owner.getExprOwner(obj).isDeleted());

    if (own.isDeleted()) {
      throw new RuntimeException("Expression depends on deleted statement: " + owner.getExprOwner(obj));
    }

    switch (own.getIrType()) {
      case AssignmentStmt: {
        return checkAssignment(obj, own);
      }
      case PhiStmt: {
        return checkPhi(obj, own);
      }
      default: {
        throw new RuntimeException("Unhandled type: " + own.getIrType());
      }
    }

  }

  // reduces if all options of phi are variable references to the same variable
  private Expression checkPhi(VariableRefLinked obj, Statement own) {
    PhiStmt cmp;
    cmp = (PhiStmt) own;
    if (cmp.getOption().isEmpty()) {
      return obj;
    }
    ArrayList<Expression> values = new ArrayList<Expression>(cmp.getOption().values());
    if (values.get(0).getIrType() != IrType.VariableRefLinked) {
      return obj;
    }
    VariableRefLinked base = (VariableRefLinked) values.get(0);
    for (int i = 1; i < values.size(); i++) {
      if (values.get(i).getIrType() != IrType.VariableRefLinked) {
        return obj;
      }
      VariableRefLinked itr = (VariableRefLinked) values.get(i);
      if (base.getReference() != itr.getReference()) {
        return obj;
      }
    }
    return new VariableRefLinked(base.getReference());
  }

  private Expression checkAssignment(VariableRefLinked obj, Statement own) {
    AssignmentStmt cmp;
    cmp = (AssignmentStmt) own;
    if ((cmp == null) || !mm.isMoveAllowed(bb.getCode(), cmp, stmt)) {
      return obj;
    }
    Set<Statement> ref = getAllReferencees(cmp);

    if (ref.size() <= 0) {
      throw new RuntimeException("impossible combination, obj is referencing cmp: " + obj + " <=> " + cmp + " :: nr " + own.getNumber() );
    }

    Expression expr = cmp.getSource();

    int cost = InliningCostCalculator.getCost(expr);
    cost *= ref.size() - 1;

    if (cost > 0) {
      return obj; // too expensive
    } else {
      return ExprCopy.copy(cmp.getSource());
    }
  }

  private Set<Statement> getAllReferencees(Assignment cmp) {
    Set<Statement> res = new HashSet<Statement>();
    KnowReferencees referencees = (KnowReferencees) kb.getEntry(KnowReferencees.class);
    for (Assignable ass : cmp.getDestination()) {
      if (ass.getIrType() == IrType.Variable) {
        res.addAll(referencees.getReferencees((Variable) ass));
      }
    }
    return res;
  }

}
