/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package reduction;


import java.util.HashSet;
import java.util.Set;

import knowledge.KnowReferencees;
import knowledge.KnowUnlinked;
import knowledge.KnowledgeBase;
import cfg.Application;
import cfg.IrType;
import cfg.basicblock.BasicBlock;
import cfg.function.Function;
import cfg.function.PrgFunction;
import cfg.statement.PhiStmt;
import cfg.statement.Statement;

/**
 * Marks phi statements using unlinked variables as deleted
 * 
 * @author urs
 * 
 */
public class UnlinkedPhiDeleter {
  private KnowReferencees kre;
  private KnowUnlinked    kul;
  private Set<PhiStmt>    removed = new HashSet<PhiStmt>();

  public static int process(PrgFunction func, KnowledgeBase base) {
    UnlinkedPhiDeleter deleter = new UnlinkedPhiDeleter(base);
    return deleter.tuwas(func);
  }

  public static int process(Application app) {
    int num = 0;
    for (Function func : app.getFunctions()) {
      if (func.getIrType() == IrType.FuncProgram) {
        num += process((PrgFunction) func, app.getKb());
      }
    }
    return num;
  }

  public UnlinkedPhiDeleter(KnowledgeBase kb) {
    super();
    kre = (KnowReferencees) kb.getEntry(KnowReferencees.class);
    kul = (KnowUnlinked) kb.getEntry(KnowUnlinked.class);
  }

  private int tuwas(PrgFunction func) {
    keep(func);
    return remove(func);
  }

  private int remove(PrgFunction func) {
    for (PhiStmt phi : removed) {
      phi.setDeleted();
    }
    return removed.size();
  }

  private void keep(PrgFunction func) {
    for (BasicBlock bb : func.getGraph().vertexSet()) {
      for (Statement stmt : bb.getCode()) {
        if (!stmt.isDeleted()) {
          if (stmt instanceof PhiStmt) {
            if (kul.hasUnlinked(stmt)) {
              remove(stmt);
            }
          }
        }
      }
    }
  }

  private void remove(Statement stmt) {
    if (!(stmt instanceof PhiStmt)) {
      throw new RuntimeException("Non-phi function depending on incomplete phi function: " + stmt);
    }
    if (removed.contains(stmt)) {
      return;
    }
    removed.add((PhiStmt) stmt);
    Set<Statement> ref = kre.getReferencees(((PhiStmt) stmt).getVarname());
    for (Statement itr : ref) {
      remove(itr);
    }
  }
}
