/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package structuring;


import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jgrapht.DirectedGraph;

import structuring.matcher.BusyWaitMatcher;
import structuring.matcher.CaseMatcher;
import structuring.matcher.CompositionMatcher;
import structuring.matcher.EndlessLoop;
import structuring.matcher.IfThenElseMatcher;
import structuring.matcher.IfThenMatcher;
import structuring.matcher.IfThenReturnMatcher;
import structuring.matcher.Pattern;
import structuring.matcher.ShortCircuitMatcher;
import structuring.matcher.WhileMatcher;
import util.NumPrint;
import util.StmtUtil;
import ast.PrgFunc;
import ast.statement.BlockStmt;
import ast.statement.Statement;
import ast.statement.VarDef;
import ast.type.Type;
import ast.variable.Variable;
import cfg.basicblock.BasicBlock;
import cfg.basicblock.BbEdge;
import cfg.function.FuncProtParam;
import cfg.function.FunctionGraph;
import cfg.function.PrgFunction;
import cfg.variable.Array;

public class Structurer {

  static public PrgFunc process(PrgFunction func) {
    PrgFunc prg = new PrgFunc(tuwas(func.getGraph(), func.getAddr()), "func" + NumPrint.toString(func.getAddr()),
        func.getReturnType());

    for (FuncProtParam param : func.getParam()) {
      Variable var = new Variable(param.name, Type.Generic);
      prg.getParam().add(var);
    }

    BlockStmt block = new BlockStmt();
    for (Array array : func.getArrays()) {
      if (array.getSize() > 1) {
        VarDef var = new VarDef(new Variable(array.getName().toString(), Type.Pointer, (int) array.getSize()));
        block.addCode(var);
      }
    }
    block.addCode(prg.getBody());
    prg.setBody(block);

    return prg;
  }

  static private Statement tuwas(FunctionGraph graph, long id) {
    // Translate content of basic blocks to AST (without jump)
    Map<BasicBlock, Statement> mapping = new HashMap<BasicBlock, Statement>();
    for (BasicBlock bb : graph.vertexSet()) {
      List<cfg.statement.Statement> stmts = bb.getCode();
      assert (StmtUtil.getNrOfStmts(stmts) >= 1);
      BlockStmt block = Translator.translate(stmts);
      stmts = stmts.subList(0, stmts.size() - 1);
      for (cfg.statement.Statement stmt : stmts) {
        stmt.setDeleted();
      }
      assert (block != null);
      mapping.put(bb, block);
    }

    Merger merger = new Merger(graph);

    while (graph.vertexSet().size() > 1) {
      List<Pattern> pattern = new ArrayList<Pattern>();

      pattern.add(new ShortCircuitMatcher());

      if (!reduce(pattern, graph, merger, mapping)) {
        break;
      }
    }

    while (graph.vertexSet().size() > 1) {
      List<Pattern> pattern = new ArrayList<Pattern>();

      pattern.add(new BusyWaitMatcher());
      pattern.add(new EndlessLoop());
      pattern.add(new CompositionMatcher());

      if (!reduce(pattern, graph, merger, mapping)) {
        break;
      }
    }

    while (graph.vertexSet().size() > 1) {
      List<Pattern> pattern = new ArrayList<Pattern>();

      pattern.add(new BusyWaitMatcher());
      pattern.add(new WhileMatcher());
      pattern.add(new IfThenReturnMatcher());
      pattern.add(new IfThenMatcher());
      pattern.add(new IfThenElseMatcher());
      pattern.add(new EndlessLoop());
      pattern.add(new CompositionMatcher());
      pattern.add(new CaseMatcher());

      if (!reduce(pattern, graph, merger, mapping)) {
        break;
      }
    }

    if (graph.vertexSet().size() > 1) {
//      System.out.println("shit, use fallback");
      Fallback.fallback(graph, mapping);
    }

    assert (graph.vertexSet().size() == 1);
    return mapping.get(graph.vertexSet().iterator().next());
  }

  private static boolean reduce(List<Pattern> pattern, DirectedGraph<BasicBlock, BbEdge> graph, Merger merger,
      Map<BasicBlock, Statement> mapping) {
    for (Pattern p : pattern) {
      for (BasicBlock v : graph.vertexSet()) {
        if (p.match(v)) { // TODO calculate cost for every match and use cheapest (can be expensive in the future if
                          // greedy algorithm is used)

          ArrayList<BasicBlock> configr = p.vertices().get(0);
          ArrayList<BbEdge> confige = p.edges().get(0);
          List<BasicBlock> mv = configr.subList(0, p.internalVertices());
          assert (confige.size() >= p.internalEdges());
          List<BbEdge> me = confige.subList(0, p.internalEdges());
          Statement code = p.getStmtCode(configr, mapping);
          BasicBlock newv = new BasicBlock(mv.get(0).getId());
          newv.addCode(p.getNewJump(configr));
          merger.merge(mv, me, newv);
          mapping.put(newv, code);
          return true;
        }
      }
    }
    return false;
  }

}
