/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package structuring;


import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import ast.Ast;
import ast.expression.CompareOp;
import ast.expression.Expression;
import ast.expression.FunctionRefUnlinked;
import ast.expression.IntegerOp;
import ast.expression.NullExpr;
import ast.expression.UnaryOp;
import ast.statement.BlockStmt;
import cfg.BooleanConstant;
import cfg.IntConstant;
import cfg.IrElement;
import cfg.IrTraverser;
import cfg.StringConstant;
import cfg.basicblock.BasicBlock;
import cfg.basicblock.BbEdge;
import cfg.expression.CallExprLinked;
import cfg.expression.CallExprPointer;
import cfg.expression.CallExprUnlinked;
import cfg.expression.CompareExpr;
import cfg.expression.IfExpr;
import cfg.expression.IntegerExpr;
import cfg.expression.UnaryExpression;
import cfg.expression.VariableKilled;
import cfg.expression.VariableRefLinked;
import cfg.expression.VariableRefUnlinked;
import cfg.function.Function;
import cfg.function.LibFunction;
import cfg.function.PrgFunction;
import cfg.function.SysFunction;
import cfg.statement.AssignmentStmt;
import cfg.statement.JumpStmt;
import cfg.statement.NopStmt;
import cfg.statement.PhiStmt;
import cfg.statement.RetStmt;
import cfg.statement.Statement;
import cfg.variable.Array;
import cfg.variable.ArrayAccess;
import cfg.variable.GlobalVariable;
import cfg.variable.SsaVariable;
import cfg.variable.StackVariable;
import cfg.variable.VariablePtrDeref;
import cfg.variable.VariablePtrOf;

public class Translator extends IrTraverser<Ast, Void> {

  public static ast.statement.BlockStmt translate(List<cfg.statement.Statement> irlist) {
    Translator visitor = new Translator();
    BlockStmt block = (BlockStmt) visitor.visit(irlist, null);
    assert (block != null);
    return block;
  }

  public static ast.expression.Expression translate(cfg.expression.Expression expression) {
    Translator visitor = new Translator();
    Expression expr = (Expression) visitor.visit(expression, null);
    assert (expr != null);
    return expr;
  }

  private Ast visit(Object obj) {
    return visit(obj, null);
  }

  @Override
  protected Ast visitBbEdge(BbEdge obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitSsaVariable(SsaVariable obj, Void param) {
    return new ast.variable.VariableRefUnlinked(obj.getName().toString());
  }

  @Override
  protected Ast visitArray(Array obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitFunction(Function obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitSysFunction(SysFunction obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitPrgFunction(PrgFunction obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitLibFunction(LibFunction obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitBasicBlock(BasicBlock obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitCollection(Collection<IrElement> obj, Void param) {
    BlockStmt block = new BlockStmt();
    for (IrElement elem : obj) {
      assert (elem instanceof Statement);
      if (!((Statement) elem).isDeleted() && !(elem instanceof JumpStmt)) {
        ast.statement.Statement stmt = (ast.statement.Statement) visit(elem);
        assert (stmt != null);
        block.addCode(stmt);
      }
    }
    return block;
  }

  @Override
  protected Ast visitCallExprLinked(CallExprLinked obj, Void param) {
    long id;
    if (obj.getFunc() instanceof PrgFunction) {
      id = ((PrgFunction) obj.getFunc()).getAddr();
    } else {
      id = -obj.getFunc().hashCode();
    }
    List<Expression> params = new ArrayList<Expression>();
    for (cfg.expression.Expression arg : obj.getParam()) {
      params.add((Expression) visit(arg));
    }
    ast.expression.CallExpr call = new ast.expression.CallExpr(new FunctionRefUnlinked(id,obj.getFunc().toString()), params);
    return call;
  }

  @Override
  protected Ast visitCallExprUnlinked(CallExprUnlinked obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitCallExprPointer(CallExprPointer obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitUnaryExpression(UnaryExpression obj, Void param) {
    UnaryOp op;
    switch( obj.getOp() ){
      case Neg: op = UnaryOp.Neg; break;
      case Not: op = UnaryOp.Not; break;
      default: throw new RuntimeException( "Unknown operator: " + obj.getOp() );
    }
    return new ast.expression.UnaryExpression((Expression)visit(obj.getExpr()), op);
  }

  @Override
  protected Ast visitVariableRefUnlinked(VariableRefUnlinked obj, Void param) {
    return new ast.variable.VariableRefUnlinked(obj.getName().toString());
  }

  @Override
  protected Ast visitVariableRefLinked(VariableRefLinked obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitVariableKilled(VariableKilled obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitBooleanConstant(BooleanConstant obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitConstant(IntConstant obj, Void param) {
    long value = obj.getValue();
    if( value > Integer.MAX_VALUE ){
      value = value - 0x100000000l;
    }
    assert( value <= Integer.MAX_VALUE );
    assert( value >= Integer.MIN_VALUE );
    return new ast.expression.IntConstant(value);
  }

  @Override
  protected Ast visitStringConstant(StringConstant obj, Void param) {
    return new ast.expression.StringConstant(obj.getContent());
  }

  @Override
  protected Ast visitVariablePtrOf(VariablePtrOf obj, Void param) {
    return new ast.expression.VariablePtrOf((ast.variable.ArrayAccess) visit(obj.getExpression()));
  }

  @Override
  protected Ast visitVariableArrayAccess(ArrayAccess obj, Void param) {
    return new ast.variable.ArrayAccess(new ast.variable.VariableRefUnlinked(obj.getBase().toString()),
        (ast.expression.Expression) visit(obj.getIndex()));
  }

  @Override
  protected Ast visitVariablePtrDeref(VariablePtrDeref obj, Void param) {
    return new ast.expression.VariablePtrDeref((Expression) visit(obj.getExpression()));
  }

  @Override
  protected Ast visitStackVariable(StackVariable obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitGlobalVariable(GlobalVariable obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitIfExpr(IfExpr obj, Void param) {
    return new ast.expression.IfExpr( (ast.expression.Expression)visit(obj.getCondition()), (ast.expression.Expression)visit(obj.getLeft()), (ast.expression.Expression)visit(obj.getRight()));
  }

  @Override
  protected Ast visitIntegerExpr(IntegerExpr obj, Void param) {
    ast.expression.IntegerOp op;
    switch (obj.getOp()) {
      case Add:
        op = IntegerOp.Add;
        break;
      case And:
        op = IntegerOp.And;
        break;
      case Div:
        op = IntegerOp.Div;
        break;
      case Mul:
        op = IntegerOp.Mul;
        break;
      case Or:
        op = IntegerOp.Or;
        break;
      case ShiftArithmeticLeft:
      case ShiftLeft:
        op = IntegerOp.ShiftLeft;
        break;
      case ShiftArithmeticRight:
      case ShiftRight:
        op = IntegerOp.ShiftRight;
        break;
      case Sub:
        op = IntegerOp.Sub;
        break;
      case Xor:
        op = IntegerOp.Xor;
        break;
      default:
        throw new RuntimeException("Not handled operand: " + obj.getOp());
    }
    return new ast.expression.ArithmeticExpr((ast.expression.Expression) visit(obj.getLeft()),
        (ast.expression.Expression) visit(obj.getRight()), op);
  }

  @Override
  protected Ast visitCompareExpr(CompareExpr obj, Void param) {
    ast.expression.CompareOp op;
    switch (obj.getOperand()) {
      case EQUAL:
        op = CompareOp.EQUAL;
        break;
      case GREATER:
        op = CompareOp.GREATER;
        break;
      case GREATER_EQUAL:
        op = CompareOp.GREATER_EQUAL;
        break;
      case LOWER:
        op = CompareOp.LOWER;
        break;
      case LOWER_EQUAL:
        op = CompareOp.LOWER_EQUAL;
        break;
      case NOT_EQUAL:
        op = CompareOp.NOT_EQUAL;
        break;
      default:
        throw new RuntimeException("Not handled operand: " + obj.getOperand());
    }
    return new ast.expression.CompareExpr((ast.expression.Expression) visit(obj.getLeft()),
        (ast.expression.Expression) visit(obj.getRight()), op);
  }

  @Override
  protected Ast visitPhiStmt(PhiStmt obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitRetStmt(RetStmt obj, Void param) {
    Expression retval;
    if (obj.getRetval().size() == 1) {
      retval = (Expression) visit(obj.getRetval().values().iterator().next());
    } else {
      assert (obj.getRetval().size() == 0);
      retval = new NullExpr();
    }
    return new ast.statement.RetStmt(retval);
  }

  @Override
  protected Ast visitJumpStmt(JumpStmt obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitNopStmt(NopStmt obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Ast visitAssignmentStmt(AssignmentStmt obj, Void param) {
    switch (obj.getDestination().size()) {
      case 0:
        return new ast.statement.CallStmt((ast.expression.CallExpr) visit(obj.getSource()));
      case 1:
        return new ast.statement.AssignmentStmt((Expression) visit(obj.getDestination().get(0)),
            (ast.expression.Expression) visit(obj.getSource()));
      default:
        throw new RuntimeException("Not yet implemented");
    }
  }

}
