/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package structuring.matcher;


import java.util.ArrayList;
import java.util.Map;

import structuring.Translator;
import util.Combinations;
import util.StmtUtil;
import ast.expression.Expression;
import ast.expression.UnaryExpression;
import ast.expression.UnaryOp;
import ast.statement.BlockStmt;
import ast.statement.IfStmt;
import ast.statement.Statement;
import cfg.basicblock.BasicBlock;
import cfg.basicblock.BbEdge;
import cfg.statement.JumpStmt;

/*
 *  0
 *  |\
 *  | 1
 *  |/
 *  2
 */

public class IfThenMatcher implements Pattern {
  private ArrayList<ArrayList<BasicBlock>> vertices = new ArrayList<ArrayList<BasicBlock>>();
  private ArrayList<ArrayList<BbEdge>>     edges    = new ArrayList<ArrayList<BbEdge>>();


  public ArrayList<ArrayList<BbEdge>> edges() {
    return edges;
  }


  public int internalEdges() {
    return 2;
  }


  public ArrayList<ArrayList<BasicBlock>> vertices() {
    return vertices;
  }


  public int internalVertices() {
    return 2;
  }

  public boolean match(BasicBlock pred) {
    ArrayList<BbEdge> outlist = new ArrayList<BbEdge>(pred.getOutlist());
    if (outlist.size() != 2) {
      return false;
    }
    Combinations combinations = new Combinations(2, outlist.size());

    while (combinations.hasNext()) {
      ArrayList<Integer> comb = combinations.next();

      BbEdge predExit = outlist.get(comb.get(0));
      BbEdge predThen = outlist.get(comb.get(1));
      BasicBlock join = predExit.getDst();
      BasicBlock then = predThen.getDst();

      if (then == pred) {
        continue;
      }

      if (then.getInlist().size() != 1) { // TODO allow this if insertion of goto is needed
        continue;
      }

      if (then.getOutlist().size() != 1) {
        continue;
      }

      BasicBlock join2 = then.getOutlist().iterator().next().getDst();

      if (join == join2) {
        addToVertices(pred, then, join);
        addToEdges(predThen, predExit);
      }
    }
    return !vertices.isEmpty();
  }

  private void addToVertices(BasicBlock predicate, BasicBlock then, BasicBlock join) {
    ArrayList<BasicBlock> list = new ArrayList<BasicBlock>(3);
    list.add(predicate);
    list.add(then);
    list.add(join);
    vertices.add(list);
  }

  private void addToEdges(BbEdge predThen, BbEdge predExit) {
    ArrayList<BbEdge> list = new ArrayList<BbEdge>(2);
    list.add(predThen);
    list.add(predExit);
    edges.add(list);
  }


  public JumpStmt getNewJump(ArrayList<BasicBlock> config) {
    return new JumpStmt(config.get(2).getId());
  }


  public Statement getStmtCode(ArrayList<BasicBlock> config, Map<BasicBlock, Statement> mapping) {
    assert (vertices.contains(config));
    assert (StmtUtil.getNrOfStmts(config.get(0).getCode()) == 1);
    Expression expr = Translator.translate(StmtUtil.toJump(config.get(0).getCode().getLast()).getExpression());
    if (isInverted(config)) {
      expr = new UnaryExpression(expr, UnaryOp.Not);
    }
    assert (config.get(1).getCode().getLast() instanceof JumpStmt);

    BlockStmt block = new BlockStmt();
    IfStmt ifs = new IfStmt(expr, mapping.get(config.get(1)));
    block.addCode(mapping.get(config.get(0)));
    block.addCode(ifs);
    return block;
  }

  private boolean isInverted(ArrayList<BasicBlock> config) {
    return StmtUtil.getEdgeValue(config.get(0), config.get(1)) == 0;
  }

}
