/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package structuring.matcher;


import java.util.ArrayList;
import java.util.Map;

import structuring.Translator;
import util.Combinations;
import util.StmtUtil;
import ast.expression.Expression;
import ast.expression.UnaryExpression;
import ast.expression.UnaryOp;
import ast.statement.BlockStmt;
import ast.statement.IfStmt;
import ast.statement.Statement;
import cfg.basicblock.BasicBlock;
import cfg.basicblock.BbEdge;
import cfg.statement.JumpStmt;

/*
 *  0
 *  |\
 *  | 1
 *  |
 *  2
 */

public class IfThenReturnMatcher implements Pattern {
  private ArrayList<ArrayList<BasicBlock>> vertices = new ArrayList<ArrayList<BasicBlock>>();
  private ArrayList<ArrayList<BbEdge>>     edges    = new ArrayList<ArrayList<BbEdge>>();


  public ArrayList<ArrayList<BbEdge>> edges() {
    return edges;
  }


  public int internalEdges() {
    return 1;
  }


  public ArrayList<ArrayList<BasicBlock>> vertices() {
    return vertices;
  }


  public int internalVertices() {
    return 2;
  }

  public boolean match(BasicBlock pred) {
    ArrayList<BbEdge> outlist = new ArrayList<BbEdge>(pred.getOutlist());
    if (outlist.size() != 2) {
      return false;
    }
    Combinations combinations = new Combinations(2, outlist.size());

    while (combinations.hasNext()) {
      ArrayList<Integer> comb = combinations.next();

      BbEdge thenEdge = outlist.get(comb.get(1));
      BasicBlock join = outlist.get(comb.get(0)).getDst();
      BasicBlock then = thenEdge.getDst();

      if (then == pred) {
        continue;
      }

      if (then.getOutlist().size() != 0) {
        continue;
      }

      if (then.getInlist().size() != 1) {
        continue;
      }

      addToVerices(pred, then, join);
      addToEdges(thenEdge);
    }
    return !vertices.isEmpty();
  }

  private void addToVerices(BasicBlock predicate, BasicBlock then, BasicBlock join) {
    ArrayList<BasicBlock> list = new ArrayList<BasicBlock>(3);
    list.add(predicate);
    list.add(then);
    list.add(join);
    vertices.add(list);
  }

  private void addToEdges(BbEdge then) {
    ArrayList<BbEdge> list = new ArrayList<BbEdge>(1);
    list.add(then);
    edges.add(list);
  }


  public JumpStmt getNewJump(ArrayList<BasicBlock> config) {
    return new JumpStmt(config.get(2).getId());
  }


  public Statement getStmtCode(ArrayList<BasicBlock> config, Map<BasicBlock, Statement> mapping) {
    assert (vertices.contains(config));
    Expression expr = Translator.translate(StmtUtil.toJump(config.get(0).getCode().getLast()).getExpression());
    if (isInverted(config)) {
      expr = new UnaryExpression(expr, UnaryOp.Not);
    }
    BlockStmt block = new BlockStmt();
    IfStmt ifs = new IfStmt(expr, mapping.get(config.get(1)));
    block.addCode(mapping.get(config.get(0)));
    block.addCode(ifs);
    return block;
  }

  private boolean isInverted(ArrayList<BasicBlock> config) {
    return StmtUtil.getEdgeValue(config.get(0), config.get(1)) == 0;
  }

}
