/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package ast.traverser;

import java.util.List;

import ast.expression.CallExpr;
import ast.expression.Expression;
import ast.statement.AssignmentStmt;
import ast.statement.BlockStmt;
import ast.statement.CallStmt;
import ast.statement.CaseStmt;
import ast.statement.DoWhile;
import ast.statement.IfStmt;
import ast.statement.NullStmt;
import ast.statement.RetStmt;
import ast.statement.Statement;
import ast.statement.VarDef;
import ast.statement.WhileStmt;
import ast.variable.Variable;

public abstract class AstStatementTraverser<P> {
  
  abstract public Expression visit( Expression expr, P param );
  abstract public Variable visit( Variable expr, P param );

  final public void visit(List<Statement> obj, P param) {
    for (int i = 0; i < obj.size(); i++) {
      Statement stmt = obj.get(i);
      stmt = visit(stmt, param);
      obj.set(i, stmt);
    }
  }

  public Statement visit(Statement obj, P param) {
    if (obj instanceof AssignmentStmt) return visitAssignmentStmt((AssignmentStmt) obj, param);
    else if (obj instanceof RetStmt) return visitRetStmt((RetStmt) obj, param);
    else if (obj instanceof DoWhile) return visitDoWhileStmt((DoWhile) obj, param);
    else if (obj instanceof WhileStmt) return visitWhileStmt((WhileStmt) obj, param);
    else if (obj instanceof BlockStmt) return visitBlockStmt((BlockStmt) obj, param);
    else if (obj instanceof IfStmt) return visitIfStmt((IfStmt) obj, param);
    else if (obj instanceof CallStmt) return visitCallStmt((CallStmt) obj, param);
    else if (obj instanceof CaseStmt) return visitCaseStmt((CaseStmt) obj, param);
    else if (obj instanceof VarDef) return visitVarDef((VarDef) obj, param);
    else if (obj instanceof NullStmt) return visitNullStmt((NullStmt) obj, param);
    else
      throw new RuntimeException("Unhandled class: " + obj.getClass());
  }

  protected Statement visitNullStmt(NullStmt obj, P param) {
    return obj;
  }

  protected Statement visitVarDef(VarDef obj, P param) {
    obj.setVariable(visit(obj.getVariable(), param));
    return obj;
  }

  protected Statement visitCallStmt(CallStmt obj, P param) {
    obj.setCall((CallExpr) visit(obj.getCall(), param));
    return obj;
  }

  protected Statement visitIfStmt(IfStmt obj, P param) {
    obj.setCondition(visit(obj.getCondition(), param));
    obj.setThenBranch(visit(obj.getThenBranch(), param));
    obj.setElseBranch(visit(obj.getElseBranch(), param));
    return obj;
  }

  protected Statement visitBlockStmt(BlockStmt obj, P param) {
    visit(obj.getBlock(), param);
    return obj;
  }

  protected Statement visitWhileStmt(WhileStmt obj, P param) {
    obj.setCondition(visit(obj.getCondition(), param));
    obj.setBody(visit(obj.getBody(), param));
    return obj;
  }

  protected Statement visitDoWhileStmt(DoWhile obj, P param) {
    obj.setBody(visit(obj.getBody(), param));
    obj.setCondition(visit(obj.getCondition(), param));
    return obj;
  }

  protected Statement visitCaseStmt(CaseStmt obj, P param) {
    obj.setCondition(visit(obj.getCondition(), param));
    for (int key : obj.getOption().keySet()) {
      Statement value = obj.getOption().get(key);
      value = visit(value, param);
      obj.getOption().put(key, value);
    }
    obj.setOther(visit(obj.getOther(), param));
    return obj;
  }

  protected Statement visitRetStmt(RetStmt obj, P param) {
    obj.setRetval(visit(obj.getRetval(), param));
    return obj;
  }

  protected Statement visitAssignmentStmt(AssignmentStmt obj, P param) {
    obj.setSource(visit(obj.getSource(), param));
    obj.setDestination(visit(obj.getDestination(), param));
    return obj;
  }

}
