/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package ast.traverser;

import graph.TestGraph;

import java.util.HashSet;
import java.util.Set;

import ast.PrgFunc;
import ast.expression.ArithmeticExpr;
import ast.expression.CallExpr;
import ast.expression.CompareExpr;
import ast.expression.Expression;
import ast.expression.IfExpr;
import ast.expression.VariablePtrDeref;
import ast.expression.VariablePtrOf;
import ast.statement.AssignmentStmt;
import ast.statement.Statement;
import ast.variable.ArrayAccess;
import ast.variable.Variable;
import ast.variable.VariableRefLinked;

public class AstVarDependency extends AstTraverser<Set<Variable>> {

  public static TestGraph<Variable> process(PrgFunc func) {
    AstVarDependency depvis = new AstVarDependency();
    depvis.visit(func, null);

    return depvis.getStmt().getExpr().getDependency();
  }

  public AstVarDependency() {
    super(new StmtVarDependency());
  }

  @Override
  public StmtVarDependency getStmt() {
    return (StmtVarDependency) super.getStmt();
  }

}

class StmtVarDependency extends AstStatementTraverser<Set<Variable>> {
  private ExprVarDependency exptrav = new ExprVarDependency();

  public final ExprVarDependency getExpr() {
    return exptrav;
  }

  @Override
  public Expression visit(Expression expr, Set<Variable> param) {
    return exptrav.visit(expr, param);
  }

  @Override
  public Variable visit(Variable expr, Set<Variable> param) {
    return exptrav.visit(expr, null);
  }

  @Override
  protected Statement visitAssignmentStmt(AssignmentStmt obj, Set<Variable> param) {
    param = new HashSet<Variable>();
    visit(obj.getSource(), param);
    exptrav.addDependencies(((VariableRefLinked) obj.getDestination()).getReference(), param);
    return obj;
  }

}

class ExprVarDependency extends AstExpressionTraverser<Set<Variable>> {

  private TestGraph<Variable> dependency = new TestGraph<Variable>();

  @Override
  public Variable visit(Variable obj, Set<Variable> param) {
    assert (!dependency.vertexSet().contains(obj));
    dependency.addVertex(obj);
    return obj;
  }

  public void addDependencies(Variable var, Set<Variable> sources) {
    for (Variable v : sources) {
      dependency.addEdge(var, v);
    }
  }

  public final TestGraph<Variable> getDependency() {
    return dependency;
  }

  @Override
  public Expression visit(Expression obj, Set<Variable> param) {
    if (param != null) {
      super.visit(obj, param);
    }
    return obj;
  }

  @Override
  protected Expression visitArrayAccess(ArrayAccess obj, Set<Variable> param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Expression visitVariablePtrOf(VariablePtrOf obj, Set<Variable> param) {
    return obj;
  }

  @Override
  protected Expression visitVariablePtrDeref(VariablePtrDeref obj, Set<Variable> param) {
    return obj;
  }

  @Override
  protected Expression visitCallExpr(CallExpr obj, Set<Variable> param) {
    return obj;
  }

  @Override
  protected Expression visitVariableRefLinked(VariableRefLinked obj, Set<Variable> param) {
    param.add(obj.getReference());
    return obj;
  }

  @Override
  protected Expression visitIfExpr(IfExpr obj, Set<Variable> param) {
    visit(obj.getLeft(), param);
    visit(obj.getRight(), param);
    return obj;
  }

  @Override
  protected Expression visitArithmeticExpr(ArithmeticExpr obj, Set<Variable> param) {
    return super.visitArithmeticExpr(obj, param);
  }

  @Override
  protected Expression visitCompareExpr(CompareExpr obj, Set<Variable> param) {
    return super.visitCompareExpr(obj, param);
  }

}
