/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg;

import cfg.expression.CompareExpr;
import cfg.expression.CompareOp;
import cfg.expression.Expression;
import cfg.expression.IntegerExpr;
import cfg.expression.IntegerOp;
import cfg.expression.VariableRefUnlinked;
import disassembler.diStorm3.OpcodeEnum;

public enum Condition {
	Overflow,
	NoOverflow,
	Below,
	AboveOrEqual,
	EqualZero,
	NotEqualNotZero,
	BelowOrEqual,
	Above,
	Sign,
	NoSign,
	Parity,
	NoParity,
	LessThan,
	GreaterThanOrEqual,
	LessThanOrEqual,
	GreaterThan;
	
	
	public static Condition parse(OpcodeEnum opcode) {
		switch(opcode){
		case JO:    case CMOVO:   case SETO:   return Overflow;
		case JNO:   case CMOVNO:  case SETNO:  return NoOverflow;
		case JB:    case CMOVB:   case SETB:   return Below;
		case JAE:   case CMOVAE:  case SETAE:  return AboveOrEqual;
		case JZ:    case CMOVZ:   case SETZ:   return EqualZero;
		case JNZ:   case CMOVNZ:  case SETNZ:  return NotEqualNotZero;
		case JBE:   case CMOVBE:  case SETBE:  return BelowOrEqual;
		case JA:    case CMOVA:   case SETA:   return Above;
		case JS:    case CMOVS:   case SETS:   return Sign;
		case JNS:   case CMOVNS:  case SETNS:  return NoSign;
		case JP:    case CMOVP:   case SETP:   return Parity;
		case JNP:   case CMOVNP:  case SETNP:  return NoParity;
		case JL:    case CMOVL:   case SETL:   return LessThan;
		case JGE:   case CMOVGE:  case SETGE:  return GreaterThanOrEqual;
		case JLE:   case CMOVLE:  case SETLE:  return LessThanOrEqual;
		case JG:    case CMOVG:   case SETG:   return GreaterThan;
		
		default: return null;
		}
	}

	public static Expression buildExpr( Condition condition ){
		VariableRefUnlinked	o = new VariableRefUnlinked(Flags.Overflow);
		VariableRefUnlinked	c = new VariableRefUnlinked(Flags.Carry);
		VariableRefUnlinked	z = new VariableRefUnlinked(Flags.Zero);
		VariableRefUnlinked	s = new VariableRefUnlinked(Flags.Sign);
		VariableRefUnlinked	p = new VariableRefUnlinked(Flags.Parity);
		IntConstant	_0 = new IntConstant(0);
		IntConstant	_1 = new IntConstant(1);
		
		switch(condition){
		case Overflow: return new CompareExpr( o, _1, CompareOp.EQUAL );
		case NoOverflow: return new CompareExpr( o, _0, CompareOp.EQUAL );
		case Below: return new CompareExpr( c, _1, CompareOp.EQUAL );
		case AboveOrEqual: return new CompareExpr( c, _0, CompareOp.EQUAL );
		case EqualZero: return new CompareExpr( z, _1, CompareOp.EQUAL );
		case NotEqualNotZero: return new CompareExpr( z, _0, CompareOp.EQUAL );
		case BelowOrEqual: return new IntegerExpr( new CompareExpr( c, _1, CompareOp.EQUAL ), new CompareExpr( z, _1, CompareOp.EQUAL ), IntegerOp.Or ); 
		case Above: return new IntegerExpr( new CompareExpr( c, _0, CompareOp.EQUAL ), new CompareExpr( z, _0, CompareOp.EQUAL ), IntegerOp.And ); 

		case Sign: return new CompareExpr( s, _1, CompareOp.EQUAL );
		case NoSign: return new CompareExpr( s, _0, CompareOp.EQUAL );
		case Parity: return new CompareExpr( p, _1, CompareOp.EQUAL );
		case NoParity: return new CompareExpr( p, _0, CompareOp.EQUAL );
		case LessThan: return new CompareExpr( s,  o, CompareOp.NOT_EQUAL );
		case GreaterThanOrEqual: return new CompareExpr( s,  o, CompareOp.EQUAL );
		case LessThanOrEqual: return new IntegerExpr( new CompareExpr( z, _1, CompareOp.EQUAL ), new CompareExpr( s, o, CompareOp.NOT_EQUAL ), IntegerOp.Or ); 
		case GreaterThan: return new IntegerExpr( new CompareExpr( z, _0, CompareOp.EQUAL ), new CompareExpr( s, o, CompareOp.EQUAL ), IntegerOp.And ); 
		
		default: throw new RuntimeException( "Unknown condition: " + condition );
		}
	}

}
