/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg;


import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import cfg.basicblock.BasicBlock;
import cfg.expression.CallExprLinked;
import cfg.expression.CallExprPointer;
import cfg.expression.CallExprUnlinked;
import cfg.expression.CompareExpr;
import cfg.expression.Expression;
import cfg.expression.IfExpr;
import cfg.expression.IntegerExpr;
import cfg.expression.UnaryExpression;
import cfg.expression.VariableKilled;
import cfg.expression.VariableRefLinked;
import cfg.expression.VariableRefUnlinked;
import cfg.function.Function;
import cfg.function.LibFunction;
import cfg.function.PrgFunction;
import cfg.statement.AssignmentStmt;
import cfg.statement.JumpStmt;
import cfg.statement.NopStmt;
import cfg.statement.PhiStmt;
import cfg.statement.RetStmt;
import cfg.statement.Statement;
import cfg.variable.Array;
import cfg.variable.ArrayAccess;
import cfg.variable.GlobalVariable;
import cfg.variable.SsaVariable;
import cfg.variable.StackVariable;
import cfg.variable.VariableAddrCalc;
import cfg.variable.VariableName;
import cfg.variable.VariablePtrDeref;
import cfg.variable.VariablePtrOf;

public class IrReplaceExprTraverser<P> {
  public void visit(IrElement obj, P param) {
    if (obj instanceof Statement) visitStatement((Statement) obj, param);
    else if (obj instanceof Expression) visitExpression((Expression) obj, param);
    else if (obj instanceof BasicBlock) visitBasicBlock((BasicBlock) obj, param);
    else if (obj instanceof Function) visitFunction((Function) obj, param);
    else if (obj instanceof SsaVariable) visitSsaVariable((SsaVariable) obj, param);
    else if (obj instanceof Array) visitArray((Array) obj, param);
    else
      throw new RuntimeException("Unhandled class: " + obj.getClass());
  }

  public void visitCollection(Collection<? extends IrElement> obj, P param) {
    for (IrElement itr : obj) {
      // assert( !(itr instanceof Expression) );
      visit(itr, param);
    }
  }

  protected void visitArray(Array obj, P param) {
  }

  protected void visitSsaVariable(SsaVariable obj, P param) {
  }

  protected void visitFunction(Function obj, P param) {
    if (obj instanceof LibFunction) visitLibFunction((LibFunction) obj, param);
    else if (obj instanceof PrgFunction) visitPrgFunction((PrgFunction) obj, param);
    else
      throw new RuntimeException("Unhandled class: " + obj.getClass());
  }

  protected void visitPrgFunction(PrgFunction obj, P param) {
    visitCollection( obj.getArrays(), param );
    visitCollection(obj.getGraph().vertexSet(), param);
  }

  protected void visitLibFunction(LibFunction obj, P param) {
  }

  protected void visitBasicBlock(BasicBlock obj, P param) {
    visitCollection(obj.getCode(), param);
  }

  protected void visitExprSet(Set<Expression> obj, P param) {
    Set<Expression> old = new HashSet<Expression>(obj);
    obj.clear();
    for (Expression itr : old) {
      obj.add(visitExpression(itr, param));
    }
  }

  protected void visitExprList(List<Expression> obj, P param) {
    for (int i = 0; i < obj.size(); i++) {
      obj.set(i, visitExpression(obj.get(i), param));
    }
  }

  protected Expression visitExpression(Expression obj, P param) {
    if (obj instanceof CompareExpr) return visitCompareExpr((CompareExpr) obj, param);
    else if (obj instanceof IntegerExpr) return visitIntegerExpr((IntegerExpr) obj, param);
    else if (obj instanceof IfExpr) return visitIfExpr((IfExpr) obj, param);
    else if (obj instanceof Memory) return visitMemory((Memory) obj, param);
    else if (obj instanceof VariableRefUnlinked) return visitVariableRefUnlinked((VariableRefUnlinked) obj, param);
    else if (obj instanceof VariableRefLinked) return visitVariableRefLinked((VariableRefLinked) obj, param);
    else if (obj instanceof VariableKilled) return visitVariableKilled((VariableKilled) obj, param);
    else if (obj instanceof IntConstant) return visitConstant((IntConstant) obj, param);
    else if (obj instanceof StringConstant) return visitStringConstant((StringConstant) obj, param);
    else if (obj instanceof BooleanConstant) return visitBooleanConstant((BooleanConstant) obj, param);
    else if (obj instanceof UnaryExpression) return visitUnaryExpression((UnaryExpression) obj, param);
    else if (obj instanceof CallExprLinked) return visitCallExprLinked((CallExprLinked) obj, param);
    else if (obj instanceof CallExprUnlinked) return visitCallExprUnlinked((CallExprUnlinked) obj, param);
    else if (obj instanceof CallExprPointer) return visitCallExprPointer((CallExprPointer) obj, param);
    else
      throw new RuntimeException("Unhandled class: " + obj.getClass());
  }

  protected Expression visitCallExprLinked(CallExprLinked obj, P param) {
    visitExprList(obj.getParam(), param);
    return obj;
  }

  protected Expression visitCallExprUnlinked(CallExprUnlinked obj, P param) {
    visitExprList(obj.getParam(), param);
    return obj;
  }

  protected Expression visitCallExprPointer(CallExprPointer obj, P param) {
    obj.setExpr(visitExpression(obj.getExpr(), param));
    visitExprList(obj.getParam(), param);
    return obj;
  }

  protected Expression visitUnaryExpression(UnaryExpression obj, P param) {
    obj.setExpr(visitExpression(obj.getExpr(), param));
    return obj;
  }

  protected Expression visitVariableRefUnlinked(VariableRefUnlinked obj, P param) {
    return obj;
  }

  protected Expression visitVariableRefLinked(VariableRefLinked obj, P param) {
    return obj;
  }

  protected Expression visitVariableKilled(VariableKilled obj, P param) {
    return obj;
  }

  protected Expression visitMemory(Memory obj, P param) {
    if (obj instanceof GlobalVariable) return visitGlobalVariable((GlobalVariable) obj, param);
    else if (obj instanceof ArrayAccess) return visitVariableArray((ArrayAccess) obj, param);
    else if (obj instanceof StackVariable) return visitStackVariable((StackVariable) obj, param);
    else if (obj instanceof VariableAddrCalc) return visitVariableAddrCalc((VariableAddrCalc) obj, param);
    else
      throw new RuntimeException("Unhandled class: " + obj.getClass());
  }

  protected BooleanConstant visitBooleanConstant(BooleanConstant obj, P param) {
    return obj;
  }

  protected Expression visitStringConstant(StringConstant obj, P param) {
    return obj;
  }

  protected IntConstant visitConstant(IntConstant obj, P param) {
    return obj;
  }

  protected Expression visitVariableAddrCalc(VariableAddrCalc obj, P param) {
    if (obj instanceof VariablePtrDeref) return visitVariablePtrDeref((VariablePtrDeref) obj, param);
    else if (obj instanceof VariablePtrOf) return visitVariablePtrOf((VariablePtrOf) obj, param);
    else
      throw new RuntimeException("Unhandled class: " + obj.getClass());
  }

  protected Expression visitVariablePtrOf(VariablePtrOf obj, P param) {
    obj.setExpression(visitExpression(obj.getExpression(), param));
    return obj;
  }

  protected Expression visitVariableArray(ArrayAccess obj, P param) {
    obj.setIndex(visitExpression(obj.getIndex(), param));
    // obj.setBase((VariableRef) visit(obj.getBase(), param));
    return obj;
  }

  protected Expression visitVariablePtrDeref(VariablePtrDeref obj, P param) {
    obj.setExpression(visitExpression(obj.getExpression(), param));
    return obj;
  }

  protected StackVariable visitStackVariable(StackVariable obj, P param) {
    return obj;
  }

  protected GlobalVariable visitGlobalVariable(GlobalVariable obj, P param) {
    return obj;
  }

  protected Expression visitIfExpr(IfExpr obj, P param) {
    obj.setCondition(visitExpression(obj.getCondition(), param));
    obj.setLeft(visitExpression(obj.getLeft(), param));
    obj.setRight(visitExpression(obj.getRight(), param));
    return obj;
  }

  protected Expression visitIntegerExpr(IntegerExpr obj, P param) {
    obj.setLeft(visitExpression(obj.getLeft(), param));
    obj.setRight(visitExpression(obj.getRight(), param));
    return obj;
  }

  protected Expression visitCompareExpr(CompareExpr obj, P param) {
    obj.setLeft(visitExpression(obj.getLeft(), param));
    obj.setRight(visitExpression(obj.getRight(), param));
    return obj;
  }

  protected void visitStatement(Statement obj, P param) {
    if (obj.isDeleted()) {
      return;
    }
    if (obj instanceof JumpStmt) visitJumpStmt((JumpStmt) obj, param);
    else if (obj instanceof PhiStmt) visitPhiStmt((PhiStmt) obj, param);
    else if (obj instanceof AssignmentStmt) visitAssignmentStmt((AssignmentStmt) obj, param);
    else if (obj instanceof NopStmt) visitNopStmt((NopStmt) obj, param);
    else if (obj instanceof RetStmt) visitRetStmt((RetStmt) obj, param);
    else
      throw new RuntimeException("Unhandled class: " + obj.getClass());
  }

  protected void visitPhiStmt(PhiStmt obj, P param) {
    visit(obj.getVarname(), param);
    for (Long bb : obj.getOption().keySet()) {
      obj.getOption().put(bb, visitExpression(obj.getOption().get(bb), param));
    }
  }

  protected void visitRetStmt(RetStmt obj, P param) {
    Map<VariableName, Expression> ret = obj.getRetval();
    for (VariableName var : ret.keySet()) {
      ret.put(var, visitExpression(ret.get(var), param));
    }
  }

  protected void visitJumpStmt(JumpStmt obj, P param) {
    obj.setExpression(visitExpression(obj.getExpression(), param));
  }

  protected void visitNopStmt(NopStmt obj, P param) {
  }

  protected void visitAssignmentStmt(AssignmentStmt obj, P param) {
    obj.setSource(visitExpression(obj.getSource(), param));
    visitCollection(obj.getDestination(), param);
  }
}
