/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg;


import java.util.Collection;
import java.util.Map;

import cfg.basicblock.BasicBlock;
import cfg.basicblock.BbEdge;
import cfg.expression.CallExprLinked;
import cfg.expression.CallExprPointer;
import cfg.expression.CallExprUnlinked;
import cfg.expression.CompareExpr;
import cfg.expression.Expression;
import cfg.expression.IfExpr;
import cfg.expression.IntegerExpr;
import cfg.expression.UnaryExpression;
import cfg.expression.VariableKilled;
import cfg.expression.VariableRefLinked;
import cfg.expression.VariableRefUnlinked;
import cfg.function.Function;
import cfg.function.LibFunction;
import cfg.function.PrgFunction;
import cfg.function.SysFunction;
import cfg.statement.AssignmentStmt;
import cfg.statement.JumpStmt;
import cfg.statement.NopStmt;
import cfg.statement.PhiStmt;
import cfg.statement.RetStmt;
import cfg.statement.Statement;
import cfg.variable.Array;
import cfg.variable.ArrayAccess;
import cfg.variable.GlobalVariable;
import cfg.variable.SsaVariable;
import cfg.variable.StackVariable;
import cfg.variable.VariableAddrCalc;
import cfg.variable.VariableName;
import cfg.variable.VariablePtrDeref;
import cfg.variable.VariablePtrOf;

public class IrTraverser<R, P> {
  @SuppressWarnings({ "unchecked" })
  public R visit(Object obj, P param) {
    if (obj instanceof Statement) return visitStatement((Statement) obj, param);
    else if (obj instanceof Expression) return visitExpression((Expression) obj, param);
    else if (obj instanceof BasicBlock) return visitBasicBlock((BasicBlock) obj, param);
    else if (obj instanceof Function) return visitFunction((Function) obj, param);
    else if (obj instanceof SsaVariable) return visitSsaVariable((SsaVariable) obj, param);
    else if (obj instanceof Array) return visitArray((Array) obj, param);
    else if (obj instanceof Collection) return visitCollection((Collection<IrElement>) obj, param);
    else
      throw new RuntimeException("Unhandled class: " + obj.getClass());
  }

  protected R visitBbEdge(BbEdge obj, P param) {
    return null;
  }

  protected R visitSsaVariable(SsaVariable obj, P param) {
    return null;
  }

  protected R visitArray(Array obj, P param) {
    return null;
  }

  protected R visitFunction(Function obj, P param) {
    if (obj instanceof LibFunction) return visitLibFunction((LibFunction) obj, param);
    else if (obj instanceof PrgFunction) return visitPrgFunction((PrgFunction) obj, param);
    else if (obj instanceof SysFunction) return visitSysFunction((SysFunction) obj, param);
    else
      throw new RuntimeException("Unhandled class: " + obj.getClass());
  }

  protected R visitSysFunction(SysFunction obj, P param) {
    return null;
  }

  protected R visitPrgFunction(PrgFunction obj, P param) {
    visit(obj.getArrays(), param);
    visit(obj.getGraph().vertexSet(), param);
    return null;
  }

  protected R visitLibFunction(LibFunction obj, P param) {
    return null;
  }

  protected R visitBasicBlock(BasicBlock obj, P param) {
    visit(obj.getCode(), param);
    return null;
  }

  protected R visitCollection(Collection<IrElement> obj, P param) {
    for (Object itr : obj) {
      visit(itr, param);
    }
    return null;
  }

  protected R visitExpression(Expression obj, P param) {
    if (obj instanceof CompareExpr) return visitCompareExpr((CompareExpr) obj, param);
    else if (obj instanceof IntegerExpr) return visitIntegerExpr((IntegerExpr) obj, param);
    else if (obj instanceof IfExpr) return visitIfExpr((IfExpr) obj, param);
    else if (obj instanceof Memory) return visitMemory((Memory) obj, param);
    else if (obj instanceof VariableRefUnlinked) return visitVariableRefUnlinked((VariableRefUnlinked) obj, param);
    else if (obj instanceof VariableRefLinked) return visitVariableRefLinked((VariableRefLinked) obj, param);
    else if (obj instanceof VariableKilled) return visitVariableKilled((VariableKilled) obj, param);
    else if (obj instanceof IntConstant) return visitConstant((IntConstant) obj, param);
    else if (obj instanceof StringConstant) return visitStringConstant((StringConstant) obj, param);
    else if (obj instanceof BooleanConstant) return visitBooleanConstant((BooleanConstant) obj, param);
    else if (obj instanceof UnaryExpression) return visitUnaryExpression((UnaryExpression) obj, param);
    else if (obj instanceof CallExprLinked) return visitCallExprLinked((CallExprLinked) obj, param);
    else if (obj instanceof CallExprUnlinked) return visitCallExprUnlinked((CallExprUnlinked) obj, param);
    else if (obj instanceof CallExprPointer) return visitCallExprPointer((CallExprPointer) obj, param);
    else
      throw new RuntimeException("Unhandled class: " + obj.getClass());
  }

  protected R visitCallExprLinked(CallExprLinked obj, P param) {
    visit(obj.getParam(), param);
    return null;
  }

  protected R visitCallExprUnlinked(CallExprUnlinked obj, P param) {
    visit(obj.getParam(), param);
    return null;
  }

  protected R visitCallExprPointer(CallExprPointer obj, P param) {
    visit(obj.getExpr(), param);
    visit(obj.getParam(), param);
    return null;
  }

  protected R visitUnaryExpression(UnaryExpression obj, P param) {
    visit(obj.getExpr(), param);
    return null;
  }

  protected R visitVariableRefUnlinked(VariableRefUnlinked obj, P param) {
    return null;
  }

  protected R visitVariableRefLinked(VariableRefLinked obj, P param) {
    return null;
  }

  protected R visitVariableKilled(VariableKilled obj, P param) {
    return null;
  }

  protected R visitMemory(Memory obj, P param) {
    if (obj instanceof GlobalVariable) return visitGlobalVariable((GlobalVariable) obj, param);
    else if (obj instanceof ArrayAccess) return visitVariableArrayAccess((ArrayAccess) obj, param);
    else if (obj instanceof StackVariable) return visitStackVariable((StackVariable) obj, param);
    else if (obj instanceof VariableAddrCalc) return visitVariableAddrCalc((VariableAddrCalc) obj, param);
    else if (obj instanceof BooleanConstant) return visitBooleanConstant((BooleanConstant) obj, param);
    else
      throw new RuntimeException("Unhandled class: " + obj.getClass());
  }

  protected R visitBooleanConstant(BooleanConstant obj, P param) {
    return null;
  }

  protected R visitConstant(IntConstant obj, P param) {
    return null;
  }

  protected R visitStringConstant(StringConstant obj, P param) {
    return null;
  }

  protected R visitVariableAddrCalc(VariableAddrCalc obj, P param) {
    if (obj instanceof VariablePtrDeref) return visitVariablePtrDeref((VariablePtrDeref) obj, param);
    else if (obj instanceof VariablePtrOf) return visitVariablePtrOf((VariablePtrOf) obj, param);
    else
      throw new RuntimeException("Unhandled class: " + obj.getClass());
  }

  protected R visitVariablePtrOf(VariablePtrOf obj, P param) {
    visit(obj.getExpression(), param);
    return null;
  }

  protected R visitVariableArrayAccess(ArrayAccess obj, P param) {
    visit(obj.getIndex(), param);
    return null;
  }

  protected R visitVariablePtrDeref(VariablePtrDeref obj, P param) {
    visit(obj.getExpression(), param);
    return null;
  }

  protected R visitStackVariable(StackVariable obj, P param) {
    return null;
  }

  protected R visitGlobalVariable(GlobalVariable obj, P param) {
    return null;
  }

  protected R visitIfExpr(IfExpr obj, P param) {
    visit(obj.getCondition(), param);
    visit(obj.getLeft(), param);
    visit(obj.getRight(), param);
    return null;
  }

  protected R visitIntegerExpr(IntegerExpr obj, P param) {
    visit(obj.getLeft(), param);
    visit(obj.getRight(), param);
    return null;
  }

  protected R visitCompareExpr(CompareExpr obj, P param) {
    visit(obj.getLeft(), param);
    visit(obj.getRight(), param);
    return null;
  }

  protected R visitStatement(Statement obj, P param) {
    if (obj.isDeleted()) {
      return null;
    }
    if (obj instanceof JumpStmt) return visitJumpStmt((JumpStmt) obj, param);
    else if (obj instanceof PhiStmt) return visitPhiStmt((PhiStmt) obj, param);
    else if (obj instanceof AssignmentStmt) return visitAssignmentStmt((AssignmentStmt) obj, param);
    else if (obj instanceof NopStmt) return visitNopStmt((NopStmt) obj, param);
    else if (obj instanceof RetStmt) return visitRetStmt((RetStmt) obj, param);
    else
      throw new RuntimeException("Unhandled class: " + obj.getClass());
  }

  protected R visitPhiStmt(PhiStmt obj, P param) {
    visit(obj.getOption().values(), param);
    visit(obj.getVarname(), param);
    return null;
  }

  protected R visitRetStmt(RetStmt obj, P param) {
    Map<VariableName, Expression> ret = obj.getRetval();
    for (VariableName var : ret.keySet()) {
      // visit(var, param);
      visit(ret.get(var), param);
    }
    return null;
  }

  protected R visitJumpStmt(JumpStmt obj, P param) {
    visit(obj.getExpression(), param);
    return null;
  }

  protected R visitNopStmt(NopStmt obj, P param) {
    return null;
  }

  protected R visitAssignmentStmt(AssignmentStmt obj, P param) {
    visit(obj.getSource(), param);
    visit(obj.getDestination(), param);
    return null;
  }
}
