/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.basicblock;


import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import util.NumPrint;
import cfg.Assignable;
import cfg.IrElement;
import cfg.IrType;
import cfg.expression.VariableRefUnlinked;
import cfg.statement.JumpStmt;
import cfg.statement.PhiStmt;
import cfg.statement.Statement;
import cfg.variable.SsaVariable;
import cfg.variable.VariableName;

public class BasicBlock implements IrElement {
  final private long               id;
  private HashMap<Object, PhiStmt> phi     = new HashMap<Object, PhiStmt>();
  private LinkedList<Statement>    code    = new LinkedList<Statement>();
  private Set<BbEdge>              inlist  = new HashSet<BbEdge>();
  private Set<BbEdge>              outlist = new HashSet<BbEdge>();

  public BasicBlock(long id) {
    super();
    this.id = id;
  }

  public Collection<PhiStmt> getPhis() {
    return phi.values();
  }

  public boolean hasPhiFor(VariableName x) {
    return phi.containsKey(x);
  }

  public void insertPhi(SsaVariable var) {
    assert (!hasPhiFor(var.getName()));
    List<Assignable> dst = new LinkedList<Assignable>();
    dst.add(var);
    PhiStmt stmt = new PhiStmt(var);
    for (BbEdge edge : inlist) {
      stmt.getOption().put(edge.getSrc().getId(), new VariableRefUnlinked(var.getName()));
    }
    phi.put(var.getName(), stmt);
    code.add(0, stmt);
  }

  public long getId() {
    return id;
  }

  private long getFirstAddr(Iterator<Statement> itr) {
    while (itr.hasNext()) {
      Statement stmt = itr.next();
      if (stmt.getOriginal() != null) {
        return stmt.getOriginal().getAddress();
      }
    }
    throw new RuntimeException("no original statement found");
  }

  public long getFirstAddr() {
    return getFirstAddr(code.iterator());
  }

  public long getLastAddr() {
    return getFirstAddr(code.descendingIterator());
  }

  public void addCode(Statement op) {
    code.add(op);
  }

  public LinkedList<Statement> getCode() {
    return code;
  }

  public void setCode(List<Statement> code) {
    this.code = new LinkedList<Statement>(code);
  }

  public void addEdgeOut(BbEdge edge) {
    assert (edge.getSrc() == this);
    outlist.add(edge);
  }

  public void addEdgeIn(BbEdge edge) {
    assert (edge.getDst() == this);
    inlist.add(edge);
  }

  public Set<BbEdge> getInlist() {
    return inlist;
  }

  public Set<BbEdge> getOutlist() {
    return outlist;
  }

  public void setOutlist(Set<BbEdge> outlist) {
    this.outlist = outlist;
  }

  public void setInlist(Set<BbEdge> inlist) {
    this.inlist = inlist;
  }

  @Override
  public String toString() {
    return NumPrint.toString(id);
  }

  public IrType getIrType() {
    return IrType.BasicBlock;
  }

  public List<? extends IrElement> getChildren() {
    return code;
  }

  /**
   * cuts the basic block @bb at address @addr into two and links everything. Returns the second basic block if a new
   * one was created.
   * 
   * @param bb
   * @param addr
   */
  public static BasicBlock splitAtAddr(BasicBlock bb, long addr) {
    if (bb.getFirstAddr() == addr) {
      return null;
    }

    BasicBlock sec = new BasicBlock(addr);
    for (int i = 0; i < bb.getCode().size(); i++) {
      if (bb.getCode().get(i).getOriginal().getAddress() == addr) {
        sec.setCode(bb.getCode().subList(i, bb.getCode().size()));
        bb.getCode().subList(i, bb.getCode().size()).clear();

        bb.addCode(new JumpStmt(0, null, sec.getId()));

        return sec;
      }
    }

    throw new RuntimeException("Address not found in BB: " + NumPrint.toString(addr));
  }

  /**
   * cuts the basic block @bb at statement nr @stmtNr into two and links everything. Returns the second basic block if a
   * new one was created.
   * 
   * @param bb
   * @param addr
   */
  public static BasicBlock splitAtStmt(BasicBlock bb, int stmtNr) {
    assert (stmtNr >= 0);
    assert (stmtNr < bb.getCode().size());

    if (stmtNr == 0) {
      return null;
    }

    List<Statement> code = bb.getCode().subList(stmtNr, bb.getCode().size());
    BasicBlock sec = new BasicBlock(code.get(0).getOriginal().getAddress());
    sec.setCode(code);
    bb.getCode().subList(stmtNr, bb.getCode().size()).clear();

    bb.addCode(new JumpStmt(0, null, sec.getId()));

    HashSet<BbEdge> nlist;
    nlist = new HashSet<BbEdge>();
    for (BbEdge edge : bb.getOutlist()) {
      nlist.add(new BbEdge(sec, edge.getDst(), edge.getNr()));
    }
    sec.setOutlist(nlist);

    nlist = new HashSet<BbEdge>();
    nlist.add(new BbEdge(bb, sec, 0));
    bb.setOutlist(nlist);

    return sec;
  }

  @Override
  public int hashCode() {
    return (int) id;
  }
}
