/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.function;


import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import util.NumPrint;
import ast.type.Type;
import cfg.IrElement;
import cfg.IrType;
import cfg.basicblock.BasicBlock;
import cfg.expression.VariableRefUnlinked;
import cfg.statement.RetStmt;
import cfg.statement.Statement;
import cfg.unlinker.HllVariable;
import cfg.variable.Array;
import cfg.variable.SsaVariable;
import cfg.variable.Variable;
import cfg.variable.VariableName;

public class PrgFunction extends Function {
  private long                     addr;
  private FunctionGraph            graph;
  private BasicBlock               entry;
  private List<BasicBlock>         exit    = new ArrayList<BasicBlock>();
  private Collection<VariableName> preserves;                              // Registers saved by this
                                                                            // function
  private List<VariableName>       returns = new ArrayList<VariableName>(); // Registers returned by this
                                                                            // function
  private int                      backupSize;
  private boolean                  useEbpAsStackAddressing;
  private int                      stacksize;
  private ArrayList<Variable>      param   = new ArrayList<Variable>(0);
  private Map<Long, Array>         arrays  = new HashMap<Long, Array>();

  // private Variable locarray = new Variable(FuncVariables.locArray, 0);

  public PrgFunction(long addr, Collection<BasicBlock> bbs) {
    super();
    this.addr = addr;
    entry = null;
    this.graph = new FunctionGraph(bbs);
    for (BasicBlock bb : graph.vertexSet()) {
      if (bb.getInlist().isEmpty()) {
        if (entry != null) {
          throw new RuntimeException("Function can have only one entry point (" + NumPrint.toString(addr) + ")");
        }
        entry = bb;
      }
      if (bb.getOutlist().isEmpty()) {
        exit.add(bb);
      }
    }
    if (entry == null) {
      throw new RuntimeException("Function needs an entry point");
    }
  }

  public void addArray(Array var) {
    arrays.put(var.getOffset(), var);
  }

  public Array getArray(long i) {
    return arrays.get(i);
  }

  public Collection<Array> getArrays() {
    return arrays.values();
  }

  public long getAddr() {
    return addr;
  }

  public ArrayList<Variable> getInternParam() {
    return param;
  }

  public void setParamCount(int paramCount) {
    assert (param.size() == 0);
    param = new ArrayList<Variable>(paramCount);
    for (int i = 0; i < paramCount; i++) {
      SsaVariable var = new SsaVariable(new HllVariable("arg" + i), 0);
      param.add(var);
      entry.insertPhi(var);
    }
  }

  public int getParamCount() {
    return param.size();
  }

  public int getStacksize() {
    return stacksize;
  }

  public void setStacksize(int stacksize) {
    this.stacksize = stacksize;
  }

  public void setBackupSize(int backupSize) {
    this.backupSize = backupSize;
  }

  public int getBackupSize() {
    return backupSize;
  }

  public boolean useEbpAsStackAddressing() {
    return useEbpAsStackAddressing;
  }

  public void setUseEbpAsStackAddressing(boolean useEbpAsStackAddressing) {
    this.useEbpAsStackAddressing = useEbpAsStackAddressing;
  }

  public Collection<VariableName> getPreserves() {
    return preserves;
  }

  public void setPreserves(Collection<VariableName> set) {
    this.preserves = set;
  }

  public void addReturnVariable(VariableName name) {
    if (returns.contains(name)) {
      for (BasicBlock bb : exit) {
        Statement stmt = bb.getCode().getLast();
        assert (stmt.getIrType() == IrType.RetStmt);
        RetStmt ret = (RetStmt) stmt;
        assert (ret.getRetval().containsKey(name));
      }
    } else {
      for (BasicBlock bb : exit) {
        Statement stmt = bb.getCode().getLast();
        assert (stmt.getIrType() == IrType.RetStmt);
        RetStmt ret = (RetStmt) stmt;
        ret.getRetval().put(name, new VariableRefUnlinked(name));
      }
      returns.add(name);
    }
  }

  @Override
  public List<VariableName> getReturns() {
    return returns;
  }

  @Override
  public String toString() {
    return "f" + Long.toHexString(getAddr());
  }

  public BasicBlock getEntry() {
    return entry;
  }

  public Collection<BasicBlock> getExit() {
    return exit;
  }

  public FunctionGraph getGraph() {
    return graph;
  }

  public IrType getIrType() {
    return IrType.FuncProgram;
  }

  public List<? extends IrElement> getChildren() {
    return new ArrayList<BasicBlock>(graph.vertexSet());
  }

  @Override
  public Type getReturnType() {
    return Type.Generic; // TODO implement it
  }

  @Override
  public List<FuncProtParam> getParam() {
    ArrayList<FuncProtParam> res = new ArrayList<FuncProtParam>(getParamCount());
    for (Variable var : getInternParam()) {
      res.add(new FuncProtParam(var.getName().toString(), Type.Generic)); // TODO implement it
    }
    return res;
  }

}
