/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.function.argument;

import cfg.IrItr;
import cfg.IrTraverser;
import cfg.expression.IntegerExpr;
import cfg.expression.IntegerOp;
import cfg.function.PrgFunction;
import cfg.matcher.VarIntopConstMatcher;
import disassembler.diStorm3.Registers;

public class FunctionArgumentCounter extends IrTraverser<Void, PrgFunction> {
  private int argMax = 0;

  static public void process(Object obj) {
    FunctionArgumentCounter replacer = new FunctionArgumentCounter();
    replacer.visit(obj, null);
  }

  @Override
  protected Void visitPrgFunction(PrgFunction obj, PrgFunction param) {
    assert (argMax == 0);
    assert (param == null);
    super.visitPrgFunction(obj, obj);
    obj.setParamCount((argMax + 3) / 4);
    argMax = 0;
    return null;
  }

  @Override
  protected Void visitIntegerExpr(IntegerExpr obj, PrgFunction param) {
    VarIntopConstMatcher matcher = new VarIntopConstMatcher();
    matcher.parse(new IrItr(obj));
    if (matcher.hasError() || !(matcher.getVar() instanceof Registers)) {
      return super.visitIntegerExpr(obj, param);
    } else {
      switch ((Registers) matcher.getVar()) {
        case EBP: {
          if (param.useEbpAsStackAddressing()) {
            doEbp(matcher.getOp(), matcher.getValue());
          }
          break;
        }
        case ESP: {
          doEsp(matcher.getOp(), matcher.getValue(), param);
          break;
        }
        default: {
          return super.visitIntegerExpr(obj, param);
        }
      }
    }
    return null;
  }

  private void doEsp(IntegerOp op, long value, PrgFunction func) {
    if (op != IntegerOp.Add) {
      throw new RuntimeException("Unexpected operation: " + op);
    }
    if (value < 0) {
      throw new RuntimeException("Unexpected value: " + value);
    }
    int argOffsetFromEsp = func.getBackupSize() + func.getStacksize() + 4;
    if (value >= argOffsetFromEsp) {
      value = value - argOffsetFromEsp;
      argMax = Math.max(argMax, (int) value + 4); // TODO: +4 depends on the data type
    }
  }

  private void doEbp(IntegerOp op, long value) {
    if ((op != IntegerOp.Add) && (op != IntegerOp.Sub)) {
      throw new RuntimeException("Unexpected operation: " + op);
    }
    if (op == IntegerOp.Sub) {
      value = -value;
    }
    if (value >= 8) {
      value = value - 8;
      assert (value >= 0);
      argMax = Math.max(argMax, (int) value + 4); // TODO: +4 depends on the data type
    }
  }

}
