/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.linker;


import java.util.Collection;
import java.util.HashMap;

import cfg.IrReplaceExprTraverser;
import cfg.basicblock.BasicBlock;
import cfg.basicblock.BbEdge;
import cfg.expression.Expression;
import cfg.expression.VariableKilled;
import cfg.expression.VariableRefUnlinked;
import cfg.function.Function;
import cfg.function.PrgFunction;
import cfg.statement.PhiStmt;
import cfg.variable.VariableName;

public class InterBbVariableLinker extends IrReplaceExprTraverser<BasicBlock> {
  private IntraBbVariableLinker           link;
  private HashMap<BasicBlock, BasicBlock> idom;
  private PrgFunction                     func;
  private boolean                         linkUnlinked;
  private boolean                         linkKilled;

  public static void link(IntraBbVariableLinker link, HashMap<BasicBlock, BasicBlock> idom, Function func,
      boolean linkUnlinked, boolean linkKilled) {
    InterBbVariableLinker linker = new InterBbVariableLinker(link, idom, linkUnlinked, linkKilled);
    linker.visit(func, null);
  }

  public InterBbVariableLinker(IntraBbVariableLinker link, HashMap<BasicBlock, BasicBlock> idom, boolean linkUnlinked,
      boolean linkKilled) {
    super();
    this.link = link;
    this.idom = idom;
    this.linkUnlinked = linkUnlinked;
    this.linkKilled = linkKilled;
  }

  @Override
  protected void visitPrgFunction(PrgFunction obj, BasicBlock param) {
    assert (func == null);
    func = obj;
    super.visitPrgFunction(obj, param);
    assert (func == obj);
    func = null;
  }

  @Override
  protected void visitBasicBlock(BasicBlock obj, BasicBlock param) {
    assert (param == null);
    super.visitBasicBlock(obj, obj);
    visitFollowingPhi(obj);
  }

  @Override
  protected void visitPhiStmt(PhiStmt obj, BasicBlock param) {
  }

  /* handle phi functions as they belong to the previous basic blocks, what they actually do */
  private void visitFollowingPhi(BasicBlock bb) {
    for (BbEdge edge : bb.getOutlist()) {
      Collection<PhiStmt> phis = edge.getDst().getPhis();
      for (PhiStmt phi : phis) {
        Expression expr = phi.getOption().get(bb.getId());
        assert (expr != null);
        expr = visitExpression(expr, bb);
        phi.getOption().put(bb.getId(), expr);
      }
    }
  }

  @Override
  protected Expression visitVariableRefUnlinked(VariableRefUnlinked obj, BasicBlock param) {
    if (linkUnlinked) {
      Expression var = getVariable(idom.get(param), obj.getName());
      assert (var != null);
      return var;
    } else {
      return super.visitVariableRefUnlinked(obj, param);
    }
  }

  @Override
  protected Expression visitVariableKilled(VariableKilled obj, BasicBlock param) {
    if (linkKilled) {
      Expression var = getVariable(idom.get(param), obj.getName());
      assert (var != null);
      return var;
    } else {
      return super.visitVariableKilled(obj, param);
    }
  }

  private Expression getVariable(BasicBlock first, VariableName name) {
    for (BasicBlock dom = first; dom != null; dom = idom.get(dom)) {
      Expression var = link.getVariable(dom, name);
      if (var != null) {
        return var;
      }
    }
    return new VariableRefUnlinked(name);
  }

}
