/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.linker;


import java.util.Collection;
import java.util.HashMap;

import knowledge.KnowKilledByFunction;
import knowledge.KnowledgeBase;
import cfg.IrReplaceExprTraverser;
import cfg.basicblock.BasicBlock;
import cfg.basicblock.BbEdge;
import cfg.expression.CallExprLinked;
import cfg.expression.CallExprPointer;
import cfg.expression.CallExprUnlinked;
import cfg.expression.Expression;
import cfg.expression.VariableKilled;
import cfg.expression.VariableRefUnlinked;
import cfg.function.Function;
import cfg.function.PrgFunction;
import cfg.statement.PhiStmt;
import cfg.variable.SsaVariable;
import cfg.variable.VariableName;

public class IntraBbVariableLinker extends IrReplaceExprTraverser<Void> {
  private HashMap<BasicBlock, RegSymTable> sym = new HashMap<BasicBlock, RegSymTable>();
  private BasicBlock                       bb  = null;
  private KnowledgeBase                    kb;
  private boolean                          linkUnlinked;
  private boolean                          linkKilled;

  public static void process(PrgFunction func, KnowledgeBase kb2, boolean linkUnlinked, boolean linkKilled) {
    IntraBbVariableLinker linker = new IntraBbVariableLinker(kb2, linkUnlinked, linkKilled);
    linker.visit(func, null);
  }

  public IntraBbVariableLinker(KnowledgeBase kb, boolean linkUnlinked, boolean linkKilled) {
    super();
    this.kb = kb;
    this.linkUnlinked = linkUnlinked;
    this.linkKilled = linkKilled;
    assert( linkUnlinked || linkKilled );
  }

  public Expression getVariable(BasicBlock bb, VariableName name) {
    if (sym.get(bb).hasDefinition(name)) {
      return sym.get(bb).getAsExpr(name);
    } else {
      return null;
    }
  }

  @Override
  protected void visitPhiStmt(PhiStmt obj, Void param) {
    visit(obj.getVarname(), param);
  }

  @Override
  protected void visitBasicBlock(BasicBlock obj, Void param) {
    assert (bb == null);
    sym.put(obj, new RegSymTable());
    bb = obj;
    super.visitBasicBlock(obj, param);
    visitFollowingPhi(obj);
    bb = null;
  }

  /* handle phi functions as they belong to the previous basic blocks, what they actually do */
  private void visitFollowingPhi(BasicBlock bb) {
    for (BbEdge edge : bb.getOutlist()) {
      Collection<PhiStmt> phis = edge.getDst().getPhis();
      for (PhiStmt phi : phis) {
        Expression expr = phi.getOption().get(bb.getId());
        assert (expr != null);
        expr = visitExpression(expr, null);
        phi.getOption().put(bb.getId(), expr);
      }
    }
  }

  @Override
  protected void visitSsaVariable(SsaVariable obj, Void param) {
    sym.get(bb).def(obj.getName(), obj);
    super.visitSsaVariable(obj, param);
  }

  @Override
  protected Expression visitVariableRefUnlinked(VariableRefUnlinked obj, Void param) {
    if (linkUnlinked) {
      SymTable sym = this.sym.get(bb);

      if (sym.hasDefinition(obj.getName())) {
        return sym.getAsExpr(obj.getName());
      } else {
        return super.visitVariableRefUnlinked(obj, param);
      }
    } else {
      return super.visitVariableRefUnlinked(obj, param);
    }
  }

  @Override
  protected Expression visitVariableKilled(VariableKilled obj, Void param) {
    if (linkKilled) {
      SymTable sym = this.sym.get(bb);

      if (sym.hasDefinition(obj.getName())) {
        return sym.getAsExpr(obj.getName());
      } else {
        return super.visitVariableKilled(obj, param);
      }
    } else {
      return super.visitVariableKilled(obj, param);
    }
  }

  @Override
  protected Expression visitCallExprLinked(CallExprLinked obj, Void param) {
    Expression res = super.visitCallExprLinked(obj, param);
    KnowKilledByFunction killedByFunction = (KnowKilledByFunction) kb.getEntry(KnowKilledByFunction.class);
    SymTable sym = this.sym.get(bb);
    Function func = obj.getFunc();
    sym.killAll(killedByFunction.getKilled(func), func);
    return res;
  }

  @Override
  protected Expression visitCallExprUnlinked(CallExprUnlinked obj, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  protected Expression visitCallExprPointer(CallExprPointer obj, Void param) {
    throw new RuntimeException("Not yet implemented");
    /*
     * SymTable sym = this.sym.get(bb); sym.killAll(Cdecl.getKilled(),null); // TODO is worst case to hard? (e.g. when
     * we know all possible functions?) return super.visitCallExprPointer(obj, param);
     */
  }

}
