/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.linker;


import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import cfg.IntConstant;
import cfg.expression.Expression;
import cfg.expression.IntegerExpr;
import cfg.expression.IntegerOp;
import cfg.expression.VariableKilled;
import cfg.expression.VariableRefLinked;
import cfg.function.Function;
import cfg.variable.Variable;
import cfg.variable.VariableName;

public class SymTable {
  private class Alias {
    VariableName lowByte;
    VariableName highByte;
    VariableName word;
    VariableName dword;

    public Alias(VariableName dword, VariableName word, VariableName highByte, VariableName lowByte) {
      super();
      this.dword = dword;
      this.word = word;
      this.highByte = highByte;
      this.lowByte = lowByte;
    }

    public int getOffset(VariableName elem) {
      if (elem == lowByte) return 0;
      else if (elem == highByte) return 1;
      else if (elem == word) return 0;
      else if (elem == dword) return 0;
      throw new RuntimeException("Not an element of this set");
    }

    public int getSize(VariableName elem) {
      if (elem == lowByte) return 1;
      else if (elem == highByte) return 1;
      else if (elem == word) return 2;
      else if (elem == dword) return 4;
      throw new RuntimeException("Not an element of this set");
    }

    /*
     * public T getSized( int size ){ switch(size){ case 1: throw new RuntimeException( "Element size is ambigous" );
     * case 2: return word; case 4: return dword; default: throw new RuntimeException( "Element size not supported: " +
     * size ); } }
     */

  }

  private Map<VariableName, Alias>    alias  = new HashMap<VariableName, Alias>();
  private Map<VariableName, Variable> def    = new HashMap<VariableName, Variable>(); // the variable was defined
  private Map<VariableName, Variable> dirty  = new HashMap<VariableName, Variable>(); // the variable was defined but a
                                                                                      // sub-var overwrote it
  private Map<VariableName, Function> killed = new HashMap<VariableName, Function>(); // the variable was killed (by
                                                                                      // function)

  public void addAlias(VariableName dword, VariableName word, VariableName highByte, VariableName lowByte) {
    assert (!alias.containsKey(dword));
    assert (!alias.containsKey(word));
    assert (!alias.containsKey(highByte));
    assert (!alias.containsKey(lowByte));

    Alias a = new Alias(dword, word, highByte, lowByte);

    alias.put(dword, a);
    alias.put(word, a);
    alias.put(highByte, a);
    alias.put(lowByte, a);
  }

  public void addAlias(VariableName dword, VariableName word) {
    assert (!alias.containsKey(dword));
    assert (!alias.containsKey(word));

    Alias a = new Alias(dword, word, null, null);

    alias.put(dword, a);
    alias.put(word, a);
  }

  public Expression getAsExpr(VariableName name) {
    assert (hasDefinition(name));
    checkInvariant();
    if (killed.containsKey(name)) {
      return new VariableKilled(name, killed.get(name));
    } else if (def.containsKey(name)) {
      Variable var = def.get(name);
      VariableRefLinked ref = createRefTo(var);

      return ref;
    } else if (dirty.containsKey(name)) {
      return reconstructDirty(name);
    } else {
      return reconstructFromBigger(name);
    }
  }

  private Expression reconstructFromBigger(VariableName name) {
    Alias a = alias.get(name);
    if (a == null) {
      return null;
    }
    int varsize = a.getSize(name);
    int offset = a.getOffset(name);

    Expression res;

    switch (varsize) {
      case 1: {
        if (def.containsKey(a.word)) {
          res = getAsExpr(a.word);
          break;
        }
      }
      case 2: {
        if (def.containsKey(a.dword)) {
          res = getAsExpr(a.dword);
          break;
        }
      }
      default: {
        throw new RuntimeException("Can not reconstruct from bigger type for size " + varsize);
      }
    }

    res = new IntegerExpr(res, new IntConstant((1 << (varsize * 8)) - 1), IntegerOp.And);
    if (offset > 0) {
      res = new IntegerExpr(res, new IntConstant(8 * offset), IntegerOp.ShiftRight);
    }
    return res;
  }

  private Expression reconstructDirty(VariableName name) {
    checkInvariant();
    assert (dirty.containsKey(name));

    Alias a = alias.get(name);
    int varsize = a.getSize(name);

    Expression res;
    switch (varsize) {
      case 4: {
        VariableRefLinked oref = createRefTo(dirty.get(name));
        VariableName defined = getDefined(a);
        if (a.getSize(defined) == 2) {
          res = new IntegerExpr(oref, new IntConstant(0xffff0000), IntegerOp.And);
          res = new IntegerExpr(res, getAsExpr(defined), IntegerOp.Or);
        } else {
          assert (a.getSize(defined) == 1);
          if (a.getOffset(defined) == 0) {
            res = new IntegerExpr(oref, new IntConstant(0xffffff00), IntegerOp.And);
            res = new IntegerExpr(res, getAsExpr(defined), IntegerOp.Or);
          } else {
            res = new IntegerExpr(oref, new IntConstant(0xffff00ff), IntegerOp.And);
            res = new IntegerExpr(res, new IntegerExpr(getAsExpr(defined), new IntConstant(8), IntegerOp.ShiftLeft),
                IntegerOp.Or);
          }
        }
        break;
      }
      case 2: {
        if (def.containsKey(a.lowByte) && def.containsKey(a.highByte)) {
          VariableRefLinked low = createRefTo(def.get(a.lowByte));
          VariableRefLinked high = createRefTo(def.get(a.highByte));
          res = new IntegerExpr(high, new IntConstant(8), IntegerOp.ShiftLeft);
          res = new IntegerExpr(res, low, IntegerOp.Or);
        } else if (def.containsKey(a.lowByte)) {
          VariableRefLinked oref = createRefTo(dirty.get(name));
          VariableRefLinked low = createRefTo(def.get(a.lowByte));
          res = new IntegerExpr(oref, new IntConstant(0xff00), IntegerOp.And);
          res = new IntegerExpr(low, res, IntegerOp.Or);
        } else if (def.containsKey(a.highByte)) {
          VariableRefLinked oref = createRefTo(dirty.get(name));
          VariableRefLinked high = createRefTo(def.get(a.highByte));
          res = new IntegerExpr(high, new IntConstant(8), IntegerOp.ShiftLeft);
          res = new IntegerExpr(res, new IntegerExpr(oref, new IntConstant(0xff), IntegerOp.And), IntegerOp.Or);
        } else {
          throw new RuntimeException("Unexpected case");
        }
        break;
      }
      default:
        throw new RuntimeException("Unsupported size: " + varsize);
    }
    return res;
  }

  private VariableName getDefined(Alias a) {
    if (def.containsKey(a.lowByte) && def.containsKey(a.highByte)) {
      return a.word;
    }
    if (def.containsKey(a.lowByte)) {
      return a.lowByte;
    }
    if (def.containsKey(a.highByte)) {
      return a.highByte;
    }
    if (def.containsKey(a.word)) {
      return a.word;
    }
    if (def.containsKey(a.dword)) {
      return a.dword;
    }
    throw new RuntimeException("Nothing defined for " + a.dword);
  }

  private VariableRefLinked createRefTo(Variable var) {
    VariableRefLinked ref = new VariableRefLinked(var);
    return ref;
  }

  public void def(VariableName name, Variable var) {
    checkInvariant();
    Alias a = alias.get(name);
    if (a != null) {
      int size = a.getSize(name);
      switch (size) {
        case 1: {
          makeDirty(a.word);
          makeDirty(a.dword);
          break;
        }
        case 2: {
          killedByBigger(a.lowByte);
          killedByBigger(a.highByte);
          makeDirty(a.dword);
          break;
        }
        case 4: {
          killedByBigger(a.lowByte);
          killedByBigger(a.highByte);
          killedByBigger(a.word);
          break;
        }
        default:
          throw new RuntimeException("Unknown type size: " + size);
      }
    }
    killed.remove(name);
    dirty.remove(name);
    def.put(name, var);
    checkInvariant();
  }

  private void killedByBigger(VariableName name) {
    def.remove(name);
    dirty.remove(name);
    checkInvariant(name);
  }

  private void makeDirty(VariableName name) {
    if (def.containsKey(name)) {
      dirty.put(name, def.get(name));
      def.remove(name);
    }
    checkInvariant(name);
  }

  public void kill(VariableName name, Function killer) {
    Alias a = alias.get(name);
    if (a != null) {
      dirty.remove(a.dword);
      def.remove(a.dword);
      killed.put(a.dword, killer);
      dirty.remove(a.word);
      def.remove(a.word);
      killed.put(a.word, killer);
      dirty.remove(a.highByte);
      def.remove(a.highByte);
      killed.put(a.highByte, killer);
      dirty.remove(a.lowByte);
      def.remove(a.lowByte);
      killed.put(a.lowByte, killer);
    } else {
      dirty.remove(name);
      def.remove(name);
      killed.put(name, killer);
    }
    checkInvariant(name);
  }

  public void killAll(Collection<? extends VariableName> names, Function killer) {
    checkInvariant();
    for (VariableName v : names) {
      kill(v, killer);
    }
    checkInvariant();
  }

  public boolean hasDefinition(VariableName name) {
    checkInvariant();

    if (killed.containsKey(name)) {
      return true;
    }
    if (dirty.containsKey(name) || def.containsKey(name)) {
      return true;
    }
    Alias a = alias.get(name);
    if (a != null) {
      int size = a.getSize(name);
      if (size <= 4) {
        if (dirty.containsKey(a.dword) || def.containsKey(a.dword)) {
          return true;
        }
      }
      if (size <= 2) {
        if (dirty.containsKey(a.word) || def.containsKey(a.word)) {
          return true;
        }
      }
      if (size == 2) {
        if (def.containsKey(a.lowByte) && def.containsKey(a.highByte)) {
          return true;
        }
      }
    }
    return false;
  }

  private void checkInvariant() {
    Set<VariableName> set = new HashSet<VariableName>();
    set.addAll(def.keySet());
    set.addAll(dirty.keySet());
    set.addAll(killed.keySet());

    Iterator<VariableName> itr = set.iterator();
    while (itr.hasNext()) {
      VariableName name = itr.next();
      Alias a = alias.get(name);
      if (a != null) {
        checkInvariant(a);
      } else {
        checkInvariant(name);
      }
    }
  }

  private void checkInvariant(Alias a) {
    checkInvariant(a.lowByte);
    checkInvariant(a.highByte);
    checkInvariant(a.word);
    checkInvariant(a.dword);

    // check if more than one alias of an variable is defined
    int n = 0;
    if (def.containsKey(a.lowByte)) n++;
    if (def.containsKey(a.highByte)) n++;
    if (def.containsKey(a.word)) n++;
    if (def.containsKey(a.dword)) n++;

    if (n > 1) {
      throw new RuntimeException("Invariant 2 does not hold for: " + a.dword);
    }
  }

  // checks if a variable is in more than one set
  private void checkInvariant(VariableName name) {
    int numdef = 0;

    if (def.containsKey(name)) numdef++;
    if (dirty.containsKey(name)) numdef++;
    if (killed.containsKey(name)) numdef++;

    if (numdef > 1) {
      throw new RuntimeException("Invariant 1 does not hold for: " + name);
    }
  }

}
