/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.parser;


import java.util.LinkedList;
import java.util.List;

import cfg.IrItr;
import cfg.IrType;
import cfg.expression.IntegerOp;
import cfg.matcher.PopMatcher;
import cfg.matcher.VarnameIntOpChangeMatcher;
import cfg.statement.Statement;
import cfg.variable.VariableName;
import disassembler.diStorm3.Registers;

enum State {
  waitespdec, // process any entry until an decrement of ESP is found
  waitpop, // process any entry until an pop is found
  expectpop, // expect pop as next instruction
  expectret, // expect ret as next instruction
  waitret, // process any entry until an ret is found
  fail, success;
}

/**
 * Parses and manipulates the prolog of an function. The variables pushed onto the stack and the stacksize for the local
 * variables is captures. The prolog is then marked as deleted.
 * 
 * @author urs
 * 
 */
public class EpilogParser {
  private LinkedList<VariableName> stack     = new LinkedList<VariableName>();
  private long                     stacksize = 0;
  private State                    state;

  public EpilogParser(List<Statement> header, List<VariableName> stack, long stacksize) {
    super();
    this.stack = new LinkedList<VariableName>(stack);
    this.stacksize = stacksize;
    if (!findNextState()) {
      state = State.waitret;
    }
    parse(header);
  }

  public boolean isSuccess() {
    return state == State.success;
  }

  private boolean findNextState() {
    if (stacksize > 0) {
      state = State.waitespdec;
      return true;
    }
    if (!stack.isEmpty()) {
      state = State.waitpop;
      return true;
    }
    return false;
  }

  private void parseSym(Statement sym) {
    switch (state) {
      case waitespdec: {
        if (isEspDec(sym)) {
          sym.setDeleted();
          if (!findNextState()) {
            state = State.expectret;
          }
        }
        break;
      }
      case waitpop:
      case expectpop: {
        if (isPop(sym)) {
          sym.setDeleted();
          if (!findNextState()) {
            state = State.expectret;
          }
        } else if (state == State.expectpop) {
          state = State.fail;
        }
        break;
      }
      case waitret:
      case expectret: {
        if (isRet(sym)) {
          state = State.success;
        } else if (state == State.expectret) {
          state = State.fail;
        }
        break;
      }
      case fail: {
        break;
      }
      case success: {
        state = State.fail;
        break;
      }
      default: {
        throw new RuntimeException("Unknown state: " + state);
      }
    }
  }

  private boolean isRet(Statement sym) {
    return sym.getIrType() == IrType.RetStmt;
  }

  private boolean isPop(Statement sym) {
    PopMatcher matcher = new PopMatcher();
    matcher.parse(new IrItr(sym));
    if (matcher.hasError()) {
      return false;
    }
    if (matcher.getVariable() != stack.peek()) {
      throw new RuntimeException("Stack problem, got " + matcher.getVariable() + " expected " + stack.peek());
    }
    stack.pop();
    return true;
  }

  private boolean isEspDec(Statement sym) {
    VarnameIntOpChangeMatcher matcher = new VarnameIntOpChangeMatcher(Registers.ESP);
    matcher.parse(new IrItr(sym));
    if (matcher.hasError()) {
      return false;
    }
    if ((matcher.getOp() != IntegerOp.Add) || (matcher.getValue() != stacksize)) {
      return false; // maybe we are in the entry bb, wait for next match
      // throw new RuntimeException("Stacksize problem");
    }
    stacksize = 0;
    return true;
  }

  private void parse(List<Statement> header) {
    for (int i = 0; i < header.size(); i++) {
      Statement stmt = header.get(i);
      parseSym(stmt);
    }
  }
}
