/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.parser;


import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Queue;
import java.util.Set;

import util.NumPrint;
import util.SymbolIndexStream;
import cfg.Application;
import cfg.IrType;
import cfg.basicblock.BasicBlock;
import cfg.expression.CallExpr;
import cfg.expression.CallExprLinked;
import cfg.expression.CallExprUnlinked;
import cfg.function.Function;
import cfg.function.LibFunction;
import cfg.statement.AssignmentStmt;
import cfg.statement.JumpStmt;
import cfg.statement.Statement;
import cfg.variable.VariableName;
import disassembler.diStorm3.DecomposedInst;
import disassembler.diStorm3.OpcodeEnum;
import disassembler.diStorm3.Registers;
import elfreader.ElfReader;

public class FunctionFactory {
  private Queue<Long>               queue    = new LinkedList<Long>();
  private HashMap<Long, BasicBlock> bbs      = new HashMap<Long, BasicBlock>();
  private ArrayList<DecomposedInst> instructions;
  private Set<Long>                 funcAddr = new HashSet<Long>();
  private int                       number   = 0;
  private ElfReader                 elfReader;
  private VariableName              retvar;
  private Application               app;

  public FunctionFactory(long entry, boolean isMain, ArrayList<DecomposedInst> instructions, ElfReader elfReader,
      Application app) {
    queue.add(entry);
    this.app = app;
    this.instructions = instructions;
    this.elfReader = elfReader;
    this.elfReader = elfReader;
    if (isMain) {
      retvar = Registers.EAX;
    } else {
      retvar = null;
    }
  }

  private boolean isAddrInRange(long addr) {
    return (addr >= instructions.get(0).getAddress())
        && (addr <= instructions.get(instructions.size() - 1).getAddress());
  }

  public void process() {
    // find and convert all basic blocks
    while (!queue.isEmpty()) {
      long addr = queue.peek();
      parseBb(addr);
      queue.remove(addr);
    }
  }

  public Set<Long> getFuncAddr() {
    return funcAddr;
  }

  public Collection<BasicBlock> getBbs() {
    return bbs.values();
  }

  /**
   * Converts all instructions from @entry to an branch, i.e. all instructions of the basic block starting at @entry. If
   * the BB ends with an jump, the destination address is handled as the start address of an basic block.
   * 
   * @param entry
   * @return
   */
  private void parseBb(long entry) {
    // System.out.println( "parse " + NumPrint.toString(entry) );

    if (isInParsedBbs(entry)) {
      return;
    }

    int idx = getIndexOf(entry);

    BasicBlock res = new BasicBlock(entry);
    OpParserImpl parser = new OpParserImpl(number, new SymbolIndexStream<DecomposedInst>(instructions, idx), elfReader,
        retvar, app);

    parser.parse();

    while (parser.hasMore()) {
      Statement op = parser.consume();
      res.addCode(op);
      OpcodeEnum opcode = op.getOriginal().getOpcode();
      if (opcode.isEndOfBb()) {
        if (opcode.isJump()) {
          JumpStmt jop = (JumpStmt) op;
          for (long addr : jop.getJmpDst()) {
            addBb(addr);
          }
        }
        if (opcode.isConditionalJump()) {
          addBb(parser.peek().getOriginal().getAddress());
        }

        bbs.put(entry, res);
        number = parser.getNumber();
        return;
      }
      if (opcode.isCall()) {
        CallExpr call = (CallExpr) ((AssignmentStmt) op).getSource();
        switch (call.getIrType()) {
          case CallExprUnlinked: {
            funcAddr.add(((CallExprUnlinked) call).getAddr());
            break;
          }
          case CallExprPointer: {
            // FIXME find solution
            break;
          }
          case CallExprLinked: {
            Function func = ((CallExprLinked) call).getFunc();
            if (func.getIrType() == IrType.FuncLibrary) {
              if (!((LibFunction) func).doesReturn()) { // if library call does not return it is the end of the bb
                bbs.put(entry, res);
                number = parser.getNumber();
                return;
              }
            }
            break;
          }
          default: {
            throw new RuntimeException("Unknown call: " + call.getIrType());
          }
        }
      }
      
      // test if next instruction is in an basic block which is already parsed
      if( parser.hasMore() ){
        long addr = parser.peek().getOriginal().getAddress();
        if( bbs.containsKey(addr) ){
          res.addCode( new JumpStmt(addr) );
          bbs.put(entry, res);
          number = parser.getNumber();
          return;
        }
      }
    }

    throw new RuntimeException("premature end of instruction stream");
  }

  private void addBb(long addr) {
    if (!isAddrInRange(addr)) { // FIXME this is probably an library call,
                                // resolve it later
      return;
    }
    if (queue.contains(addr)) {
      return;
    }
    if (!isInParsedBbs(addr)) {
      queue.add(addr);
    }
  }

  private boolean isInParsedBbs(long addr) {
    for (BasicBlock bb : bbs.values()) {
      if ((addr >= bb.getFirstAddr()) && (addr <= bb.getLastAddr())) {
        BasicBlock sec = BasicBlock.splitAtAddr(bb, addr);
        if (sec != null) {
          bbs.put(addr, sec);
        }
        return true;
      }
    }
    return false;
  }

  private int getIndexOf(long addr) {
    if (!isAddrInRange(addr)) {
      throw new RuntimeException("Address out of bounds: " + NumPrint.toString(addr));
    }
    int left = 0;
    int right = instructions.size() - 1;

    while (left < right) {
      int mid = (left + right) / 2; // Compute mid point.
      if (addr < instructions.get(mid).getAddress()) {
        right = mid; // repeat search in bottom half.
      } else if (addr > instructions.get(mid).getAddress()) {
        left = mid + 1; // Repeat search in top half.
      } else {
        return mid; // Found it. return position
      }
    }
    throw new RuntimeException("Address not found in instructions");
  }

}
