/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.parser;


import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import util.SymbolStream;
import cfg.Application;
import cfg.Assignable;
import cfg.Condition;
import cfg.Flags;
import cfg.IntConstant;
import cfg.IrType;
import cfg.expression.CallExpr;
import cfg.expression.CallExprLinked;
import cfg.expression.CallExprPointer;
import cfg.expression.CallExprUnlinked;
import cfg.expression.Expression;
import cfg.expression.IfExpr;
import cfg.expression.IntegerExpr;
import cfg.expression.IntegerOp;
import cfg.expression.UnaryExpression;
import cfg.expression.UnaryOp;
import cfg.expression.VariableRef;
import cfg.expression.VariableRefUnlinked;
import cfg.function.Function;
import cfg.function.LibFunction;
import cfg.function.system.SystemFunctions;
import cfg.statement.AssignmentStmt;
import cfg.statement.JumpStmt;
import cfg.statement.NopStmt;
import cfg.statement.RetStmt;
import cfg.statement.Statement;
import cfg.variable.ArrayAccess;
import cfg.variable.GlobalVariable;
import cfg.variable.SsaVariable;
import cfg.variable.StackVariable;
import cfg.variable.Variable;
import cfg.variable.VariableName;
import cfg.variable.VariablePtrDeref;
import disassembler.diStorm3.DecomposedInst;
import disassembler.diStorm3.Operand.OperandType;
import disassembler.diStorm3.Registers;
import elfreader.ElfReader;
import elfreader.ElfRelocationReader;

public class OpParserImpl extends OperationParser {
  private VariableName      retvar;
  private Map<Long, String> libmap;
  protected Application     app;

  public OpParserImpl(int number, SymbolStream<DecomposedInst> stream, ElfReader elfReader, VariableName retvar,
      Application app) {
    super(number, stream, elfReader);
    this.app = app;
    this.retvar = retvar;
    this.libmap = ElfRelocationReader.getFuncNames(elfReader);
  }

  private AssignmentStmt newAss(DecomposedInst inst, Expression src) {
    Assignable left;

    if (inst.mOperands[0].getType() == OperandType.Reg) {
      left = new SsaVariable(Registers.getRegisterByIndex(inst.mOperands[0].getIndex()), getNumber());
    } else {
      left = (Assignable) parseOperand(inst.mOperands[0], inst);
    }

    LinkedList<Assignable> dst = new LinkedList<Assignable>();
    dst.add(left);
    dst.addAll(createDefFlagVariables(inst.getOpcode()));

    return new AssignmentStmt(getNumber(), inst, dst, src);
  }

  private CallExpr newCall(Function func) {
    return newCall(func, new LinkedList<Expression>());
  }

  private CallExpr newCall(Function func, Expression param1) {
    LinkedList<Expression> param = new LinkedList<Expression>();
    param.add(param1);
    return newCall(func, param);
  }

  private CallExpr newCall(Function func, List<Expression> param) {
    CallExpr call = new CallExprLinked(func);
    call.setParam(param);
    return call;
  }

  private NopStmt newNop(DecomposedInst inst) {
    return new NopStmt(getNumber(), inst);
  }

  protected Statement parseTest(DecomposedInst inst) {
    assert (inst.mOperands.length == 2);
    // assert( inst.mOperands[0].getType() == OperandType.Reg ); //TODO
    // change to all needed types
    // assert( inst.mOperands[1].getType() == OperandType.Reg ); //TODO
    // change to all needed types

    assert (createUsedFlagsVariables(inst.getOpcode()).size() == 0);
    Expression expr;

    expr = new IntegerExpr(parseOperand(inst.mOperands[0], inst), parseOperand(inst.mOperands[1], inst), IntegerOp.And);

    return new AssignmentStmt(getNumber(), inst, new LinkedList<Assignable>(createDefFlagVariables(inst.getOpcode())),
        expr);
  }

  protected Statement parseMov(DecomposedInst inst) {
    assert (inst.mOperands.length == 2);
    return newAss(inst, parseOperand(inst.mOperands[1], inst));
  }

  @Override
  protected Statement parseMovs(DecomposedInst inst) {
    // TODO implement it
    /*
     * something like: 'memcpy' ( op0, op1, opsize*'ECX', 'D' ) // protoype: dst, src, size (bytes), downwards 'ESI' =
     * 'ESI' +/- opsize*'ECX' 'EDI' = 'EDI' +/- opsize*'ECX' 'ECX' = 0
     */
    if (inst.getSize() != 2) { // hack to determine if there is a REP prefix
      throw new RuntimeException("Not yet implemented");
    }
    assert (inst.mOperands.length == 2);
    Expression dst = parseOperand(inst.mOperands[0], inst);
    Expression src = parseOperand(inst.mOperands[1], inst);

    if (!((src.getIrType() == IrType.VariablePtrDeref) && (dst.getIrType() == IrType.VariablePtrDeref))) {
      throw new RuntimeException("Not yet implemented");
    }

    VariablePtrDeref dstd = (VariablePtrDeref) dst;
    VariablePtrDeref srcd = (VariablePtrDeref) src;

    dst = dstd.getExpression();
    src = srcd.getExpression();

    // FIXME replace constant with pointer size
    Expression size = new IntegerExpr(new VariableRefUnlinked(Registers.ECX), new IntConstant(4), IntegerOp.Mul);

    LinkedList<Assignable> assdst = new LinkedList<Assignable>();

    assdst.add(new SsaVariable(Registers.ECX, getNumber()));
    assdst.add(new SsaVariable(Registers.EDI, getNumber()));
    assdst.add(new SsaVariable(Registers.ESI, getNumber()));

    CallExprLinked call = new CallExprLinked(SystemFunctions.funcMemcpy);
    call.getParam().add(dst);
    call.getParam().add(src);
    call.getParam().add(size);
    call.getParam().add(new VariableRefUnlinked(Flags.Direction));

    return new AssignmentStmt(getNumber(), inst, assdst, call);
  }

  @Override
  protected Statement parseMovsd(DecomposedInst inst) {
    assert (inst.mOperands.length == 2);

    { // Check if it is really the sse instruction since a string instruction with the same name exists
      Expression op1 = parseOperand(inst.mOperands[0], inst);
      Expression op2 = parseOperand(inst.mOperands[1], inst);

      VariableRefUnlinked ref = null;
      if (op1.getIrType() == IrType.VariableRefUnlinked) {
        ref = (VariableRefUnlinked) op1;
      } else if (op2.getIrType() == IrType.VariableRefUnlinked) {
        ref = (VariableRefUnlinked) op2;
      }
      assert (ref != null);
      assert (((Registers) ref.getName()).isMmx());
    }

    return parseMov(inst);
  }

  @Override
  protected Statement parseCmovx(DecomposedInst inst) {
    assert (inst.mOperands.length == 2);

    Expression cond = Condition.buildExpr(Condition.parse(inst.getOpcode()));

    IfExpr expr = new IfExpr(cond, parseOperand(inst.mOperands[1], inst), parseOperand(inst.mOperands[0], inst));

    return newAss(inst, expr);
  }

  @Override
  protected Statement parseMovsx(DecomposedInst inst) { // TODO propagate type
    // information
    assert (inst.mOperands.length == 2);
    return newAss(inst, parseOperand(inst.mOperands[1], inst));
  }

  @Override
  protected Statement parseMovzx(DecomposedInst inst) { // TODO propagate type
    // information
    assert (inst.mOperands.length == 2);
    return newAss(inst, parseOperand(inst.mOperands[1], inst));
  }

  @Override
  protected Statement parseLea(DecomposedInst inst) {
    assert (inst.mOperands.length == 2);
    Expression var = parseOperand(inst.mOperands[1], inst);
    if (var instanceof VariablePtrDeref) {
      var = ((VariablePtrDeref) var).getExpression();
    } else if (var instanceof ArrayAccess) {
      assert (false);
      // var = new IntegerExpr(((ArrayAccess) var).getBase(), ((ArrayAccess) var).getIndex(), IntegerOp.Add);

    } else if (var instanceof StackVariable) {
      var = new IntegerExpr(new VariableRefUnlinked(Registers.ESP), new IntConstant(((StackVariable) var).getOffset()),
          IntegerOp.Add);
    } else if (var instanceof GlobalVariable) {
      // var = new VariablePtrOf(var); // TODO check if correct for DS:12345
      assert (((GlobalVariable) var).getSegment() == Registers.DS);
      var = new IntConstant(((GlobalVariable) var).getAddress());
    } else {
      throw new RuntimeException("not yet implemented: " + var + " (" + inst + ")");
    }
    return newAss(inst, var);
  }

  @Override
  protected Statement parseCmp(DecomposedInst inst) {
    assert (inst.mOperands.length == 2);
    Expression expr = new IntegerExpr(parseOperand(inst.mOperands[0], inst), parseOperand(inst.mOperands[1], inst),
        IntegerOp.Sub);
    return new AssignmentStmt(getNumber(), inst, new LinkedList<Assignable>(createDefFlagVariables(inst.getOpcode())),
        expr);
  }

  @Override
  protected Statement parsePush(DecomposedInst inst) {
    assert (inst.mOperands.length == 1);
    return new AssignmentStmt(getNumber(), inst, new LinkedList<Assignable>(), newCall(SystemFunctions.push,
        parseOperand(inst.mOperands[0], inst)));
  }

  @Override
  protected Statement parsePop(DecomposedInst inst) {
    return newAss(inst, newCall(SystemFunctions.pop));
  }

  @Override
  protected Statement parseLeave(DecomposedInst inst) {
    return newNop(inst);
  }

  private Statement parseIntegerOperation(DecomposedInst inst, IntegerOp op) {
    assert (inst.mOperands.length == 2);
    Expression lop = parseOperand(inst.mOperands[0], inst);
    Expression rop = parseOperand(inst.mOperands[1], inst);
    Expression expr = new IntegerExpr(lop, rop, op);
    return newAss(inst, expr);
  }

  private Statement parseIntegerOperationCarry(DecomposedInst inst, IntegerOp op) { // TODO
                                                                                    // verify
                                                                                    // correctness
    assert (inst.mOperands.length == 2);
    Expression lop = parseOperand(inst.mOperands[0], inst);
    Expression rop = parseOperand(inst.mOperands[1], inst);
    Expression expr = new IntegerExpr(lop, rop, op);
    expr = new IntegerExpr(expr, new VariableRefUnlinked(Flags.Carry), op);
    return newAss(inst, expr);
  }

  @Override
  protected Statement parseNeg(DecomposedInst inst) {
    assert (inst.mOperands.length == 1);
    Expression op = parseOperand(inst.mOperands[0], inst);
    Expression expr = new UnaryExpression(op, UnaryOp.Neg);
    return newAss(inst, expr);
  }

  @Override
  protected Statement parseInc(DecomposedInst inst) {
    assert (inst.mOperands.length == 1);
    Expression op = parseOperand(inst.mOperands[0], inst);
    Expression expr = new IntegerExpr(op, new IntConstant(1), IntegerOp.Add);
    return newAss(inst, expr);
  }

  @Override
  protected Statement parseDec(DecomposedInst inst) {
    assert (inst.mOperands.length == 1);
    Expression op = parseOperand(inst.mOperands[0], inst);
    Expression expr = new IntegerExpr(op, new IntConstant(1), IntegerOp.Sub);
    return newAss(inst, expr);
  }

  @Override
  protected Statement parseXmul(DecomposedInst inst) {
    // TODO: differentiate between MUL and IMUL
    switch (inst.mOperands.length) {
      case 1: {
        VariableRefUnlinked op = (VariableRefUnlinked) parseOperand(inst.mOperands[0], inst);
        assert ((op.getName() == Registers.EAX) || (op.getName() == Registers.EBP) || (op.getName() == Registers.EBX)
            || (op.getName() == Registers.ECX) || (op.getName() == Registers.EDI) || (op.getName() == Registers.EDX)
            || (op.getName() == Registers.ESI) || (op.getName() == Registers.ESP)); // FIXME extend it to all types
        Expression expr = new IntegerExpr(new VariableRefUnlinked(Registers.EAX), op, IntegerOp.Mul);
        LinkedList<Assignable> sideEffects = new LinkedList<Assignable>();
        sideEffects.add(new SsaVariable(Registers.EDX, getNumber())); // FIXME replace it with correct implementation
        sideEffects.add(new SsaVariable(Registers.EAX, getNumber()));
        sideEffects.addAll(createDefFlagVariables(inst.getOpcode()));
        return new AssignmentStmt(getNumber(), inst, sideEffects, expr);
      }
      case 2: {
        Expression expr = new IntegerExpr(parseOperand(inst.mOperands[0], inst), parseOperand(inst.mOperands[1], inst),
            IntegerOp.Mul);
        return newAss(inst, expr);
      }
      case 3: {
        Expression expr = new IntegerExpr(parseOperand(inst.mOperands[1], inst), parseOperand(inst.mOperands[2], inst),
            IntegerOp.Mul);
        return newAss(inst, expr);
      }
    }
    throw new RuntimeException("not yet implmented: " + inst);
  }

  @Override
  protected Statement parseXdiv(DecomposedInst inst) {
    // TODO: differentiate between MUL and IMUL
    switch (inst.mOperands.length) {
      case 1: {
        VariableRefUnlinked op = (VariableRefUnlinked) parseOperand(inst.mOperands[0], inst);
        assert ((op.getName() == Registers.EAX) || (op.getName() == Registers.EBP) || (op.getName() == Registers.EBX)
            || (op.getName() == Registers.ECX) || (op.getName() == Registers.EDI) || (op.getName() == Registers.EDX)
            || (op.getName() == Registers.ESI) || (op.getName() == Registers.ESP)); // FIXME
                                                                                    // extend
                                                                                    // it
                                                                                    // to
                                                                                    // all
                                                                                    // types
        Expression expr = new IntegerExpr(new VariableRefUnlinked(Registers.EAX), op, IntegerOp.Div);
        Collection<VariableRef> used = createUsedFlagsVariables(inst.getOpcode());
        LinkedList<Assignable> sideEffects = new LinkedList<Assignable>();
        sideEffects.add(new SsaVariable(Registers.EDX, getNumber())); // FIXME
        // replace it
        // with
        // correct
        // implementation
        sideEffects.add(new SsaVariable(Registers.EAX, getNumber()));
        sideEffects.addAll(createDefFlagVariables(inst.getOpcode()));
        used.add(new VariableRefUnlinked(Registers.EDX)); // FIXME replace it
                                                          // with correct
                                                          // implementation
        return new AssignmentStmt(getNumber(), inst, sideEffects, expr);
      }
      case 2: {
      }
      case 3: {
      }
    }
    throw new RuntimeException("not yet implmented: " + inst);
  }

  @Override
  protected Statement parseCdq(DecomposedInst inst) {
    assert (inst.mOperands.length == 0);
    LinkedList<Assignable> dst = new LinkedList<Assignable>();
    dst.add(new SsaVariable(Registers.EDX, getNumber()));
    dst.add(new SsaVariable(Registers.EAX, getNumber()));
    return new AssignmentStmt(getNumber(), inst, dst, newCall(SystemFunctions.castInt64, new VariableRefUnlinked(
        Registers.EAX)));
  }

  @Override
  protected Statement parseOr(DecomposedInst inst) {
    return parseIntegerOperation(inst, IntegerOp.Or);
  }

  @Override
  protected Statement parseAnd(DecomposedInst inst) {
    return parseIntegerOperation(inst, IntegerOp.And);
  }

  @Override
  protected Statement parseAdd(DecomposedInst inst) {
    return parseIntegerOperation(inst, IntegerOp.Add);
  }

  @Override
  protected Statement parseAdc(DecomposedInst inst) {
    return parseIntegerOperationCarry(inst, IntegerOp.Add);
  }

  @Override
  protected Statement parseXor(DecomposedInst inst) {
    if (inst.mOperands[0].equals(inst.mOperands[1])) {
      return newAss(inst, new IntConstant(0));
    }
    return parseIntegerOperation(inst, IntegerOp.Xor);
  }

  @Override
  protected Statement parseNot(DecomposedInst inst) {
    assert (inst.mOperands.length == 1);
    UnaryExpression expr = new UnaryExpression(parseOperand(inst.mOperands[0], inst), UnaryOp.Not);
    return newAss(inst, expr);
  }

  @Override
  protected Statement parseSub(DecomposedInst inst) {
    return parseIntegerOperation(inst, IntegerOp.Sub);
  }

  @Override
  protected Statement parseSbb(DecomposedInst inst) {
    return parseIntegerOperationCarry(inst, IntegerOp.Sub);
  }

  @Override
  protected Statement parseSar(DecomposedInst inst) {
    return parseIntegerOperation(inst, IntegerOp.ShiftArithmeticRight);
  }

  @Override
  protected Statement parseShr(DecomposedInst inst) {
    return parseIntegerOperation(inst, IntegerOp.ShiftRight);
  }

  @Override
  protected Statement parseSal(DecomposedInst inst) {
    return parseIntegerOperation(inst, IntegerOp.ShiftArithmeticLeft);
  }

  @Override
  protected Statement parseShl(DecomposedInst inst) {
    return parseIntegerOperation(inst, IntegerOp.ShiftLeft);
  }

  @Override
  protected Statement parseNop(DecomposedInst inst) {
    return newNop(inst);
  }

  @Override
  protected Statement parseSetX(DecomposedInst inst) {
    assert (inst.mOperands.length == 1);
    // Collection<VariableRefUnlinked> src =
    // createUsedFlagsVariables(inst.getOpcode());
    // assert (src.size() == 1);
    return newAss(inst, Condition.buildExpr(Condition.parse(inst.getOpcode())));
  }

  @Override
  protected Statement parseCondJmp(DecomposedInst inst) {
    return new JumpStmt(getNumber(), inst, Condition.buildExpr(Condition.parse(inst.getOpcode())), stream.peek()
        .getAddress(), inst.getConstantValue());
  }

  @Override
  protected Statement parseJmp(DecomposedInst inst) {
    Expression expr = parseOperand(inst.mOperands[0], inst);

    if (expr instanceof IntConstant) {
      return new JumpStmt(getNumber(), inst, ((IntConstant) expr).getValue());
    } else {
      long jmpTableBase;
      Expression index;

      VariablePtrDeref der;
      if (expr instanceof VariablePtrDeref) {
        der = (VariablePtrDeref) expr;
      } else {
        // check if last stmt was the address calculation
        if ((next instanceof AssignmentStmt) && (expr instanceof VariableRef)) {
          AssignmentStmt stmt = (AssignmentStmt) next;
          VariableRef var = (VariableRef) expr;
          List<Assignable> dst = stmt.getDestination();
          if ((dst.size() == 1) && (dst.iterator().next() instanceof Variable)
              && ((Variable) dst.iterator().next()).getName().equals(var.getName())) {
            if (stmt.getSource() instanceof VariablePtrDeref) {
              der = (VariablePtrDeref) stmt.getSource();
            } else {
              throw new RuntimeException("Last expression was not an jump table access: " + next);
            }
          } else {
            throw new RuntimeException("Last assignment was not a jump switch: " + next);
          }
        } else {
          throw new RuntimeException("Last statement was not an assignment: " + next);
        }
      }

      {
        IntegerExpr baseExpr = (IntegerExpr) der.getExpression();
        if (baseExpr.getOp() != IntegerOp.Add) {
          throw new RuntimeException("Unexpected operation: " + baseExpr.getOp());
        }
        IntConstant baseAddr = (IntConstant) baseExpr.getRight();
        IntegerExpr idxExpr = (IntegerExpr) baseExpr.getLeft();
        if (idxExpr.getOp() != IntegerOp.Mul) {
          throw new RuntimeException("Unexpected operation: " + idxExpr.getOp());
        }
        IntConstant addrSize = (IntConstant) idxExpr.getRight();
        index = idxExpr.getLeft();
        if (addrSize.getValue() != 4) {
          throw new RuntimeException("Unexpected addr size: " + addrSize.getValue());
        }
        jmpTableBase = baseAddr.getValue();
      }

      ArrayList<Long> jmptable = readJumpTable(jmpTableBase);
      return new JumpStmt(getNumber(), inst, index, jmptable);
    }
  }

  private ArrayList<Long> readJumpTable(long baseAddr) {
    ArrayList<Long> res = new ArrayList<Long>();
    elfReader.seek(baseAddr);
    while (true) {
      long addr = elfReader.getInt();
      // if( elfReader.isText(addr) ){
      if ((addr & 0xfff00000) == 0x08000000) { // FIXME change to always
        // working method
        res.add(addr);
      } else {
        break;
      }
    }

    return res;
  }

  @Override
  protected Statement parseCall(DecomposedInst inst) {
    Expression expr = parseOperand(inst.mOperands[0], inst);
    CallExpr func;
    if (expr instanceof IntConstant) {
      long addr = ((IntConstant) expr).getValue();
      String funcName = libmap.get(addr);
      if (funcName != null) {
        LibFunction libfunc = app.getDynlib().getFunction(funcName);
        func = new CallExprLinked(libfunc);
      } else {
        func = new CallExprUnlinked(addr);
      }
    } else {
      func = new CallExprPointer(expr);
    }
    return new AssignmentStmt(getNumber(), inst, new LinkedList<Assignable>(), func);
  }

  @Override
  protected Statement parseRet(DecomposedInst inst) {
    Map<VariableName, Expression> ret = new HashMap<VariableName, Expression>();
    if (retvar != null) {
      ret.put(retvar, new VariableRefUnlinked(retvar));
    }
    return new RetStmt(getNumber(), inst, ret);
  }

  @Override
  protected Statement parseHlt(DecomposedInst inst) {
    return new AssignmentStmt(getNumber(), inst, new LinkedList<Assignable>(), newCall(SystemFunctions.halt));
  }

}
