/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package cfg.parser;


import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import util.NumPrint;
import util.SymbolStream;
import cfg.Flags;
import cfg.IntConstant;
import cfg.expression.Expression;
import cfg.expression.IntegerExpr;
import cfg.expression.IntegerOp;
import cfg.expression.VariableRef;
import cfg.expression.VariableRefUnlinked;
import cfg.statement.Statement;
import cfg.variable.GlobalVariable;
import cfg.variable.SsaVariable;
import cfg.variable.VariablePtrDeref;
import disassembler.diStorm3.DecomposedInst;
import disassembler.diStorm3.OpcodeEnum;
import disassembler.diStorm3.Operand;
import disassembler.diStorm3.Registers;
import elfreader.ElfReader;

public abstract class OperationParser implements SymbolStream<Statement> {

  protected SymbolStream<DecomposedInst> stream;
  Statement                              next;
  private int                            number;
  protected ElfReader                    elfReader;

  public OperationParser(int number, SymbolStream<DecomposedInst> stream, ElfReader elfReader) {
    super();
    this.number = number;
    this.stream = stream;
    this.elfReader = elfReader;
  }

  public int getNumber() {
    return number;
  }

  public void parse() {
    if (!stream.hasMore()) {
      next = null;
      return;
    }
    // String pretty = stream.getNext().getPretty();
    // System.out.println(pretty);

    /*
     * if( stream.getNext().getOpcode().isEndOfBb() ){ next = parseEndOfBb( stream.consume() ); return; }
     */

    switch (stream.peek().getOpcode()) {
      case ADD:
        next = parseAdd(stream.consume());
        break;
      case ADC:
        next = parseAdc(stream.consume());
        break;
      case SUB:
        next = parseSub(stream.consume());
        break;
      case SBB:
        next = parseSbb(stream.consume());
        break;
      case NEG:
        next = parseNeg(stream.consume());
        break;
      case INC:
        next = parseInc(stream.consume());
        break;
      case DEC:
        next = parseDec(stream.consume());
        break;
      case SAL:
        next = parseSal(stream.consume());
        break;
      case SHL:
        next = parseShl(stream.consume());
        break;
      case SAR:
        next = parseSar(stream.consume());
        break;
      case SHR:
        next = parseShr(stream.consume());
        break;
      case IMUL:
        next = parseXmul(stream.consume());
        break;
      case MUL:
        next = parseXmul(stream.consume());
        break;
      case IDIV:
        next = parseXdiv(stream.consume());
        break;
      case DIV:
        next = parseXdiv(stream.consume());
        break;

      case AND:
        next = parseAnd(stream.consume());
        break;
      case OR:
        next = parseOr(stream.consume());
        break;
      case XOR:
        next = parseXor(stream.consume());
        break;
      case NOT:
        next = parseNot(stream.consume());
        break;
      case CDQ:
        next = parseCdq(stream.consume());
        break;

      case MOV:
        next = parseMov(stream.consume());
        break;
      case TEST:
        next = parseTest(stream.consume());
        break;
      case CMP:
        next = parseCmp(stream.consume());
        break;
      case LEA:
        next = parseLea(stream.consume());
        break;
      case PUSH:
        next = parsePush(stream.consume());
        break;
      case POP:
        next = parsePop(stream.consume());
        break;

      case MOVSX:
        next = parseMovsx(stream.consume());
        break;
      case MOVZX:
        next = parseMovzx(stream.consume());
        break;
      case MOVSD:
        next = parseMovsd(stream.consume());
        break;

      case MOVS:
        next = parseMovs(stream.consume());
        break;

      case CMOVA:
        next = parseCmovx(stream.consume());
        break;
      case CMOVAE:
        next = parseCmovx(stream.consume());
        break;
      case CMOVB:
        next = parseCmovx(stream.consume());
        break;
      case CMOVBE:
        next = parseCmovx(stream.consume());
        break;
      case CMOVZ:
        next = parseCmovx(stream.consume());
        break;
      case CMOVG:
        next = parseCmovx(stream.consume());
        break;
      case CMOVGE:
        next = parseCmovx(stream.consume());
        break;
      case CMOVL:
        next = parseCmovx(stream.consume());
        break;
      case CMOVLE:
        next = parseCmovx(stream.consume());
        break;
      case CMOVNZ:
        next = parseCmovx(stream.consume());
        break;
      case CMOVNO:
        next = parseCmovx(stream.consume());
        break;
      case CMOVNS:
        next = parseCmovx(stream.consume());
        break;
      case CMOVNP:
        next = parseCmovx(stream.consume());
        break;
      case CMOVO:
        next = parseCmovx(stream.consume());
        break;
      case CMOVP:
        next = parseCmovx(stream.consume());
        break;
      case CMOVS:
        next = parseCmovx(stream.consume());
        break;

      case LEAVE:
        next = parseLeave(stream.consume());
        break;
      case NOP:
        next = parseNop(stream.consume());
        break;
      case RET:
        next = parseRet(stream.consume());
        break;
      case HLT:
        next = parseHlt(stream.consume());
        break;
      case CALL:
        next = parseCall(stream.consume());
        break;
      case JMP:
        next = parseJmp(stream.consume());
        break;

      case JA:
        next = parseCondJmp(stream.consume());
        break;
      case JAE:
        next = parseCondJmp(stream.consume());
        break;
      case JB:
        next = parseCondJmp(stream.consume());
        break;
      case JBE:
        next = parseCondJmp(stream.consume());
        break;
      case JZ:
        next = parseCondJmp(stream.consume());
        break;
      case JG:
        next = parseCondJmp(stream.consume());
        break;
      case JGE:
        next = parseCondJmp(stream.consume());
        break;
      case JL:
        next = parseCondJmp(stream.consume());
        break;
      case JLE:
        next = parseCondJmp(stream.consume());
        break;
      case JNZ:
        next = parseCondJmp(stream.consume());
        break;
      case JNO:
        next = parseCondJmp(stream.consume());
        break;
      case JNS:
        next = parseCondJmp(stream.consume());
        break;
      case JNP:
        next = parseCondJmp(stream.consume());
        break;
      case JO:
        next = parseCondJmp(stream.consume());
        break;
      case JP:
        next = parseCondJmp(stream.consume());
        break;
      case JS:
        next = parseCondJmp(stream.consume());
        break;

      case SETA:
        next = parseSetX(stream.consume());
        break;
      case SETAE:
        next = parseSetX(stream.consume());
        break;
      case SETB:
        next = parseSetX(stream.consume());
        break;
      case SETBE:
        next = parseSetX(stream.consume());
        break;
      case SETZ:
        next = parseSetX(stream.consume());
        break;
      case SETG:
        next = parseSetX(stream.consume());
        break;
      case SETGE:
        next = parseSetX(stream.consume());
        break;
      case SETL:
        next = parseSetX(stream.consume());
        break;
      case SETLE:
        next = parseSetX(stream.consume());
        break;
      case SETNZ:
        next = parseSetX(stream.consume());
        break;
      case SETNO:
        next = parseSetX(stream.consume());
        break;
      case SETNS:
        next = parseSetX(stream.consume());
        break;
      case SETNP:
        next = parseSetX(stream.consume());
        break;
      case SETO:
        next = parseSetX(stream.consume());
        break;
      case SETP:
        next = parseSetX(stream.consume());
        break;
      case SETS:
        next = parseSetX(stream.consume());
        break;

      // case JCXZ: next = parseJcxz( stream.consume() ); break;
      // case JECXZ: next = parseJecxz( stream.consume() ); break;
      // case JRCXZ: next = parseJrcxz( stream.consume() ); break;

      default:
        DecomposedInst inst = stream.peek();
        throw new RuntimeException("Opcode not yet supported: " + inst + " at " + NumPrint.toString(inst.getAddress()));
    }

    number++;
  }

  abstract protected Statement parseMov(DecomposedInst inst);

  abstract protected Statement parseMovsx(DecomposedInst inst);

  abstract protected Statement parseMovzx(DecomposedInst inst);

  abstract protected Statement parseTest(DecomposedInst inst);

  abstract protected Statement parseCmp(DecomposedInst inst);

  abstract protected Statement parseAnd(DecomposedInst inst);

  abstract protected Statement parseOr(DecomposedInst inst);

  abstract protected Statement parseXor(DecomposedInst inst);

  abstract protected Statement parseNot(DecomposedInst inst);

  abstract protected Statement parseAdd(DecomposedInst inst);

  abstract protected Statement parseAdc(DecomposedInst inst);

  abstract protected Statement parseSub(DecomposedInst inst);

  abstract protected Statement parseSbb(DecomposedInst inst);

  abstract protected Statement parseSar(DecomposedInst inst);

  abstract protected Statement parseShr(DecomposedInst inst);

  abstract protected Statement parseSal(DecomposedInst inst);

  abstract protected Statement parseShl(DecomposedInst inst);

  abstract protected Statement parseLea(DecomposedInst inst);

  abstract protected Statement parsePush(DecomposedInst inst);

  abstract protected Statement parsePop(DecomposedInst inst);

  abstract protected Statement parseLeave(DecomposedInst inst);

  abstract protected Statement parseNop(DecomposedInst inst);

  abstract protected Statement parseRet(DecomposedInst inst);

  abstract protected Statement parseHlt(DecomposedInst inst);

  abstract protected Statement parseCondJmp(DecomposedInst inst);

  abstract protected Statement parseCall(DecomposedInst inst);

  abstract protected Statement parseNeg(DecomposedInst inst);

  abstract protected Statement parseInc(DecomposedInst inst);

  abstract protected Statement parseDec(DecomposedInst inst);

  abstract protected Statement parseJmp(DecomposedInst inst);

  abstract protected Statement parseXmul(DecomposedInst inst);

  abstract protected Statement parseXdiv(DecomposedInst inst);

  abstract protected Statement parseCdq(DecomposedInst inst);

  abstract protected Statement parseSetX(DecomposedInst inst);

  abstract protected Statement parseCmovx(DecomposedInst inst);

  abstract protected Statement parseMovsd(DecomposedInst inst);

  abstract protected Statement parseMovs(DecomposedInst inst);

  protected List<SsaVariable> createDefFlagVariables(OpcodeEnum opcode) {
    List<SsaVariable> res = new LinkedList<SsaVariable>();
    for (Flags flag : Flags.values()) {
      if (opcode.defFlag(flag)) {
        res.add(new SsaVariable(flag, getNumber()));
      }
    }
    return res;
  }

  protected Collection<VariableRef> createUsedFlagsVariables(OpcodeEnum opcode) {
    List<VariableRef> res = new LinkedList<VariableRef>();
    for (Flags flag : Flags.values()) {
      if (opcode.useFlag(flag)) {
        res.add(new VariableRefUnlinked(flag));
      }
    }
    return res;
  }

  static public Expression parseOperand(Operand op, DecomposedInst instruction) {
    switch (op.getType()) {
      case Reg: {
        return new VariableRefUnlinked(Registers.getRegisterByIndex(op.getIndex()));
      }
      case Disp: {
        return new GlobalVariable(Registers.getDisplacementReg(op.getIndex()), instruction.mDisp.getDisplacement());
      }
      case Imm: {
        return new IntConstant(instruction.mImm.getImm());
      }
      case Smem: {
        /*
         * if( Registers.getRegisterByIndex(op.getIndex()) == Registers.ESP ){ return new
         * StackVariable(instruction.mDisp.getDisplacement()); } else {
         */
        IntegerExpr addr = new IntegerExpr(new VariableRefUnlinked(Registers.getRegisterByIndex(op.getIndex())),
            new IntConstant(instruction.mDisp.getDisplacement()), IntegerOp.Add);
        return new VariablePtrDeref(addr);
        // }
      }
      case Mem: {
        Expression left = new IntegerExpr(new VariableRefUnlinked(Registers.getRegisterByIndex(op.getIndex())),
            new IntConstant(Math.max(instruction.getScale(), 1)), // FIXME make it correct, how is multiplicator of 1
                                                                  // defined?
            IntegerOp.Mul);

        Expression right = new IntConstant(instruction.mDisp.getDisplacement());
        if (instruction.getBase() < Registers.values().length) {
          right = new IntegerExpr(new VariableRefUnlinked(Registers.getRegisterByIndex(instruction.getBase())), right,
              IntegerOp.Add);
        }

        Expression expr = new IntegerExpr(left, right, IntegerOp.Add);

        return new VariablePtrDeref(expr);
      }
      case Pc: {
        return new IntConstant(instruction.mImm.getImm() + instruction.mDisp.getDisplacement()
            + instruction.getAddress() + instruction.getSize());
      }
      default:
        throw new RuntimeException("Operand parser not yet implemented: " + instruction);
    }
  }

  public Statement peek() {
    return next;
  }

  public Statement consume() {
    Statement res = next;
    parse();
    return res;
  }

  public boolean hasMore() {
    return next != null;
  }

}
