/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package codewriter;

import java.io.FileNotFoundException;
import java.io.PrintStream;

import util.NumPrint;
import ast.Ast;
import ast.Function;
import ast.LibFuncs;
import ast.PrgFunc;
import ast.Program;
import ast.expression.ArithmeticExpr;
import ast.expression.BooleanConstant;
import ast.expression.CallExpr;
import ast.expression.CompareExpr;
import ast.expression.Expression;
import ast.expression.FunctionRef;
import ast.expression.FunctionRefLinked;
import ast.expression.FunctionRefUnlinked;
import ast.expression.IfExpr;
import ast.expression.IntConstant;
import ast.expression.IntegerOp;
import ast.expression.NullConstExpr;
import ast.expression.StringConstant;
import ast.expression.UnaryExpression;
import ast.expression.VariablePtrDeref;
import ast.expression.VariablePtrOf;
import ast.statement.AssignmentStmt;
import ast.statement.BlockStmt;
import ast.statement.CallStmt;
import ast.statement.CaseStmt;
import ast.statement.DoWhile;
import ast.statement.IfStmt;
import ast.statement.NullStmt;
import ast.statement.RetStmt;
import ast.statement.Statement;
import ast.statement.VarDef;
import ast.statement.WhileStmt;
import ast.traverser.AstExpressionTraverser;
import ast.traverser.AstStatementTraverser;
import ast.traverser.AstTraverser;
import ast.traverser.ReturnChecker;
import ast.type.Type;
import ast.variable.ArrayAccess;
import ast.variable.Variable;
import ast.variable.VariableRefLinked;
import ast.variable.VariableRefUnlinked;

public class CodeWriter extends AstTraverser<Writer> {
  public CodeWriter(InfoWriter info) {
    super(new CodeWriterStatement(info));
  }

  public static void write( Ast prg, InfoWriter info, String filename ){
    CodeWriter writer = new CodeWriter(info);
    try {
      writer.visit(prg, new Writer(new PrintStream(filename)));
    } catch (FileNotFoundException e) {
      e.printStackTrace();
    }
  }
  
  @Override
  protected Ast visitProgram(Program obj, Writer param) {
    param.wr("public class " + obj.getOutputName() + " {");
    param.nl();
    param.incIndent();
    super.visitProgram(obj, param);
    param.decIndent();
    param.wr("}");
    param.nl();
    return obj;
  }

  @Override
  protected Ast visitPrgFunc(PrgFunc obj, Writer param) {
    param.wr("public ");
    param.wr(getTypeString(obj.getReturnType()) + " " + obj.getName() + "( ");
    for (int i = 0; i < obj.getParam().size(); i++) {
      if (i > 0) {
        param.wr(", ");
      }
      getStmt().visit(obj.getParam().get(i), param);
    }
    param.wr(") ");

    visit(obj.getBody(), param);
    param.nl();
    return obj;
  }

  static public String getTypeString(Type type) {
    switch (type) {
      case Integer: {
        return "int";
      }
      case Pointer: {
        return "Pointer<Integer>";
      }
      case Void: {
        return "void";
      }
      default: {
        return type + "?";
      }
    }
  }

}

class CodeWriterStatement extends AstStatementTraverser<Writer> {
  private CodeWriterExpression exptrav = new CodeWriterExpression();
  private InfoWriter info;

  public CodeWriterStatement(InfoWriter info) {
    this.info = info;
  }

  @Override
  public Expression visit(Expression expr, Writer param) {
    return exptrav.visit(expr, param);
  }

  @Override
  public Variable visit(Variable expr, Writer param) {
    return exptrav.visit(expr, param);
  }

  private void printInfo(Statement obj, Writer param) {
    if (info != null) {
      param.wr("\t//");
      param.wr(info.get(obj));
    }
  }

  @Override
  protected Statement visitNullStmt(NullStmt obj, Writer param) {
    param.wr(";");
    param.nl();
    return obj;
  }

  @Override
  protected Statement visitVarDef(VarDef obj, Writer param) {
    visit(obj.getVariable(), param);
    if (obj.getVariable().getSize() > 1) {
      // this are static initialized arrays
      assert (obj.getVariable().getType() == Type.Pointer);
      param.wr(" = new Pointer<Integer>( ");
      param.wr(obj.getVariable().getSize());
      param.wr(" )");
    } else {
      // needed because java is not so smart as I am (for switch statements)
      param.wr(" = ");
      param.wr(obj.getVariable().getType() == Type.Pointer ? "null" : "0");
    }
    param.wr(";");
    printInfo(obj, param);
    param.nl();
    return obj;
  }

  @Override
  protected Statement visitCallStmt(CallStmt obj, Writer param) {
    visit(obj.getCall(), param);
    param.wr(";");
    printInfo(obj, param);
    param.nl();
    return obj;
  }

  @Override
  protected Statement visitRetStmt(RetStmt obj, Writer param) {
    param.wr("return ");
    visit(obj.getRetval(), param);
    param.wr(";");
    printInfo(obj, param);
    param.nl();
    return obj;
  }

  @Override
  protected Statement visitAssignmentStmt(AssignmentStmt obj, Writer param) {
    visit(obj.getDestination(), param);
    param.wr(" = ");
    visit(obj.getSource(), param);
    param.wr(";");
    printInfo(obj, param);
    param.nl();
    return obj;
  }

  @Override
  protected Statement visitDoWhileStmt(DoWhile obj, Writer param) {
    param.wr("do ");
    visit(obj.getBody(), param);
    param.wr("while( ");
    visit(obj.getCondition(), param);
    param.wr(" )");
    param.wr(";");
    printInfo(obj, param);
    param.nl();
    return obj;
  }

  @Override
  protected Statement visitWhileStmt(WhileStmt obj, Writer param) {
    param.wr("while( ");
    visit(obj.getCondition(), param);
    param.wr(" )");
//    param.nl();
    visit(obj.getBody(), param);
    printInfo(obj, param);
    return obj;
  }

  @Override
  protected Statement visitBlockStmt(BlockStmt obj, Writer param) {
    param.wr("{");
    printInfo(obj, param);
    // param.wr( " // " + NumPrint.toString(obj.getId()));
    param.nl();
    param.incIndent();
    visit(obj.getBlock(), param);
    param.decIndent();
    param.wr("}");
    param.nl();
    return obj;
  }

  @Override
  protected Statement visitIfStmt(IfStmt obj, Writer param) {
    param.wr("if( ");
    visit(obj.getCondition(), param);
    param.wr(") ");
    visit(obj.getThenBranch(), param);
    if (!(obj.getElseBranch() instanceof NullStmt)) {
      param.wr("else ");
      visit(obj.getElseBranch(), param);
    }
    printInfo(obj, param);
    param.nl();
    return obj;
  }

  @Override
  protected Statement visitCaseStmt(CaseStmt obj, Writer param) {
    param.wr("switch( ");
    visit(obj.getCondition(), param);
    param.wr(" ){");
    printInfo(obj, param);
    param.nl();
    param.incIndent();
    for (int op : obj.getOption().keySet()) {
      param.wr("case ");
      param.wr(NumPrint.toString(op));
      param.wr(": {");
      param.nl();
      param.incIndent();
      visit(obj.getOption().get(op), param);
      if( ReturnChecker.comesBack(obj.getOption().get(op)) ){
        param.wr("break;");
        param.nl();
      }
      param.decIndent();
      param.wr("}");
      param.nl();
    }
    if (!(obj.getOther() instanceof NullStmt)) {
      param.wr("default:");
      visit(obj.getOther(), param);
    }
    param.decIndent();
    param.wr("}");
    param.nl();
    return obj;
  }

}

class CodeWriterExpression extends AstExpressionTraverser<Writer> {
  public void emitFuncCallParam(CallExpr obj, Writer param) {
    param.wr("(");
    for (int i = 0; i < obj.getParam().size(); i++) {
      if (i > 0) {
        param.wr(", ");
      }
      visit(obj.getParam().get(i), param);
    }
    param.wr(")");
  }

  @Override
  public Variable visit(Variable obj, Writer param) {
    param.wr(CodeWriter.getTypeString(obj.getType()));
    param.wr(" ");
    param.wr(obj.getName());
    return obj;
  }

  @Override
  protected Expression visitCallExpr(CallExpr obj, Writer param) {
    if (obj.getFunction() instanceof FunctionRefLinked) {
      Function func = ((FunctionRefLinked) obj.getFunction()).getFunc();

      if ((func == LibFuncs.ptrNew) || (func == LibFuncs.ptrCopy)) {
        param.wr("new Pointer<Integer>");
        emitFuncCallParam(obj, param);
      } else if (func == LibFuncs.ptrDestroy) {
        visit(obj.getParam().get(0), param);
        param.wr(".free()");
      } else if (func == LibFuncs.ptrGetValue) {
        visit(obj.getParam().get(0), param);
        param.wr(".getValue( ");
        visit(obj.getParam().get(1), param);
        param.wr(" )");
      } else if (func == LibFuncs.ptrSetValue) {
        visit(obj.getParam().get(0), param);
        param.wr(".setValue( ");
        visit(obj.getParam().get(1), param);
        param.wr(", ");
        visit(obj.getParam().get(2), param);
        param.wr(" )");
      } else {
        visit(obj.getFunction(), param);
        emitFuncCallParam(obj, param);
      }
    } else {
      visit(obj.getFunction(), param);
      emitFuncCallParam(obj, param);
    }

    return obj;
  }

  @Override
  protected FunctionRef visitFunctionRefLinked(FunctionRefLinked obj, Writer param) {
    param.wr(obj.getFunc().getName());
    return obj;
  }

  @Override
  protected FunctionRef visitFunctionRefUnlinked(FunctionRefUnlinked obj, Writer param) {
//    System.out.println("Unlinked function: " + obj );
    param.wr("unlinked_function_at_" + NumPrint.toString(obj.getAddr()) + "(" + obj.getTag() + ")");
    return obj;
  }

  @Override
  protected Expression visitUnaryExpression(UnaryExpression obj, Writer param) {
    param.wr("(");
    param.wr(obj.getOp());
    visit(obj.getExpr(), param);
    param.wr(")");
    return obj;
  }

  @Override
  protected Expression visitVariableRefUnlinked(VariableRefUnlinked obj, Writer param) {
    param.wr("'" + obj.getName().toString() + "'");
    return obj;
  }

  @Override
  protected Expression visitVariableRefLinked(VariableRefLinked obj, Writer param) {
    param.wr(obj.getReference().getName());
    return obj;
  }

  @Override
  protected Expression visitBooleanConstant(BooleanConstant obj, Writer param) {
    param.wr(obj.getValue());
    return obj;
  }

  @Override
  protected Expression visitNullConstExpr(NullConstExpr obj, Writer param) {
    param.wr("null");
    return obj;
  }

  @Override
  protected Expression visitConstant(IntConstant obj, Writer param) {
    param.wr(obj.getValue());
    return obj;
  }

  @Override
  protected Expression visitStringConstant(StringConstant obj, Writer param) {
    param.wr("\"" + obj.getContent() + "\"");
    return obj;
  }

  @Override
  protected Expression visitIfExpr(IfExpr obj, Writer param) {
    param.wr("( ");
    visit(obj.getCondition(), param);
    param.wr(" ? ");
    visit(obj.getLeft(), param);
    param.wr(" : ");
    visit(obj.getRight(), param);
    param.wr(" )");
    return obj;
  }

  @Override
  protected Expression visitArithmeticExpr(ArithmeticExpr obj, Writer param) {
    param.wr("(");
    visit(obj.getLeft(), param);
    param.wr(" ");
    param.wr(getOpStr(obj.getOp()));
    param.wr(" ");
    visit(obj.getRight(), param);
    param.wr(")");
    return obj;
  }

  private Object getOpStr(IntegerOp op) {
    switch (op) {
      case And:
        return "&";
      case Or:
        return "|";
      case Xor:
        return "^";
      case Add:
        return "+";
      case Sub:
        return "-";
      case Mul:
        return "*";
      case Div:
        return "/";
      case ShiftRight:
        return ">>";
      case ShiftLeft:
        return "<<";
      default:
        throw new RuntimeException("Not yet implemneted: " + op);
    }
  }

  @Override
  protected Expression visitCompareExpr(CompareExpr obj, Writer param) {
    param.wr("(");
    visit(obj.getLeft(), param);
    param.wr(" ");
    param.wr(obj.getOperand());
    param.wr(" ");
    visit(obj.getRight(), param);
    param.wr(")");
    return obj;
  }

  @Override
  protected Expression visitArrayAccess(ArrayAccess obj, Writer param) {
    visit(obj.getBase(), param);
    param.wr("[");
    visit(obj.getIndex(), param);
    param.wr("]");
    return obj;
  }

  @Override
  protected Expression visitVariablePtrOf(VariablePtrOf obj, Writer param) {
    param.wr("&");
    visit(obj.getVar(), param);
    return obj;
  }

  @Override
  protected Expression visitVariablePtrDeref(VariablePtrDeref obj, Writer param) {
    param.wr("*");
    visit(obj.getExpr(), param);
    return obj;
  }

}
