/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package dataflow;

import graph.BaseGraph;
import graph.SimpleEdge;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.jgrapht.DirectedGraph;

import ast.PrgFunc;
import ast.expression.Expression;
import ast.statement.AssignmentStmt;
import ast.statement.CallStmt;
import ast.statement.CaseStmt;
import ast.statement.DoWhile;
import ast.statement.IfStmt;
import ast.statement.NullStmt;
import ast.statement.RetStmt;
import ast.statement.Statement;
import ast.statement.VarDef;
import ast.statement.WhileStmt;
import ast.traverser.AstStatementTraverser;
import ast.variable.Variable;

/*
 * Creates an CFG from an AST
 */
abstract public class GraphBuilder<T> extends AstStatementTraverser<Void> {
  private DirectedGraph<DfVertex<T>, SimpleEdge<DfVertex<T>>> graph;
  private DfVertex<T>                                         last;
  private Map<DfVertex<T>, Statement>                         invmapping = new HashMap<DfVertex<T>, Statement>();
  private Map<Statement, DfVertex<T>>                         mapping    = new HashMap<Statement, DfVertex<T>>();

  public DirectedGraph<DfVertex<T>, SimpleEdge<DfVertex<T>>> build(PrgFunc func) {
    graph = new BaseGraph<DfVertex<T>, SimpleEdge<DfVertex<T>>>();

    DfVertex<T> vinit = createVertex(func);
    graph.addVertex(vinit);
    visit(func.getBody(), null);

    return graph;
  }

  public final DirectedGraph<DfVertex<T>, SimpleEdge<DfVertex<T>>> getGraph() {
    return graph;
  }

  public final Map<DfVertex<T>, Statement> getInvMapping() {
    return invmapping;
  }

  public final Map<Statement, DfVertex<T>> getMapping() {
    return mapping;
  }

  private void addVertex(Statement obj, DfVertex<T> v) {
    graph.addVertex(v);
    invmapping.put(v, obj);
    mapping.put(obj, v);
  }

  private void addEdge(DfVertex<T> src, DfVertex<T> dst) {
    if ((src != null) && (dst != null)) {
      SimpleEdge<DfVertex<T>> e = new SimpleEdge<DfVertex<T>>(src, dst);
      graph.addEdge(e.getSrc(), e.getDst(), e);
    }
  }

  abstract protected DfVertex<T> createVertex();

  abstract protected DfVertex<T> createVertex(PrgFunc obj);

  @Override
  protected Statement visitNullStmt(NullStmt obj, Void param) {
    DfVertex<T> v = createVertex(obj);
    addVertex(obj, v);
    addEdge(last, v);
    last = v;
    return obj;
  }

  abstract protected DfVertex<T> createVertex(NullStmt obj);

  @Override
  protected Statement visitVarDef(VarDef obj, Void param) {
    DfVertex<T> v = createVertex(obj);
    addVertex(obj, v);
    addEdge(last, v);
    last = v;
    return obj;
  }

  abstract protected DfVertex<T> createVertex(VarDef obj);

  @Override
  protected Statement visitCallStmt(CallStmt obj, Void param) {
    DfVertex<T> v = createVertex(obj);
    addVertex(obj, v);
    addEdge(last, v);
    last = v;
    return obj;
  }

  abstract protected DfVertex<T> createVertex(CallStmt obj);

  @Override
  protected Statement visitAssignmentStmt(AssignmentStmt obj, Void param) {
    DfVertex<T> v = createVertex(obj);
    addVertex(obj, v);
    addEdge(last, v);
    last = v;
    return obj;
  }

  abstract protected DfVertex<T> createVertex(AssignmentStmt obj);

  @Override
  protected Statement visitRetStmt(RetStmt obj, Void param) {
    DfVertex<T> v = createVertex(obj);
    addVertex(obj, v);
    addEdge(last, v);
    last = null;
    return obj;
  }

  abstract protected DfVertex<T> createVertex(RetStmt obj);

  @Override
  protected Statement visitIfStmt(IfStmt obj, Void param) {
    DfVertex<T> head = createVertex(obj);
    addVertex(obj, head);
    addEdge(last, head);

    last = head;
    visit(obj.getThenBranch(), null);
    DfVertex<T> tv = last;

    last = head;
    visit(obj.getElseBranch(), null);
    DfVertex<T> ev = last;

    DfVertex<T> join = createVertex();
    graph.addVertex(join);

    addEdge(tv, join);
    addEdge(ev, join);
    last = join;

    return obj;
  }

  abstract protected DfVertex<T> createVertex(IfStmt obj);

  @Override
  protected Statement visitCaseStmt(CaseStmt obj, Void param) {
    DfVertex<T> head = createVertex(obj);
    addVertex(obj, head);
    addEdge(last, head);

    DfVertex<T> join = createVertex();
    graph.addVertex(join);

    Set<Statement> branches = new HashSet<Statement>(obj.getOption().values());
    branches.add(obj.getOther());

    for (Statement branch : branches) {
      last = head;
      visit(branch, null);
      DfVertex<T> tv = last;
      addEdge(tv, join);
    }

    last = join;

    return obj;
  }

  abstract protected DfVertex<T> createVertex(CaseStmt obj);

  @Override
  protected Statement visitWhileStmt(WhileStmt obj, Void param) {
    DfVertex<T> join = createVertex(obj);
    addVertex(obj, join);
    addEdge(last, join);
    last = join;

    visit(obj.getBody(), null);

    DfVertex<T> foot = createVertex();
    graph.addVertex(foot);
    addEdge(last, foot);
    addEdge(foot, join);

    last = foot;

    return obj;
  }

  abstract protected DfVertex<T> createVertex(WhileStmt obj);

  @Override
  protected Statement visitDoWhileStmt(DoWhile obj, Void param) {
    DfVertex<T> join = createVertex();
    graph.addVertex(join);
    addEdge(last, join);
    last = join;

    visit(obj.getBody(), null);

    DfVertex<T> foot = createVertex(obj);
    addVertex(obj, foot);
    addEdge(last, foot);
    addEdge(foot, join);

    last = foot;

    return obj;
  }

  abstract protected DfVertex<T> createVertex(DoWhile obj);

  @Override
  public Expression visit(Expression expr, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

  @Override
  public Variable visit(Variable expr, Void param) {
    throw new RuntimeException("Not yet implemented");
  }

}
