/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package dataflow.copyprop;

import java.util.Map;

import ast.expression.Expression;
import ast.statement.AssignmentStmt;
import ast.statement.NullStmt;
import ast.statement.Statement;
import ast.traverser.AstExpressionTraverser;
import ast.traverser.AstStatementTraverser;
import ast.traverser.AstTraverser;
import ast.variable.Variable;
import ast.variable.VariableRefLinked;

public class CopyReplacer extends AstStatementTraverser<Map<Variable, Variable>> {
  private ExprCopyReplacer                        expr    = new ExprCopyReplacer();
  private Map<Statement, Map<Variable, Variable>> defset;
  private int                                     changes = 0;

  public CopyReplacer(Map<Statement, Map<Variable, Variable>> defset) {
    this.defset = defset;
  }

  public static int process(Statement ast, Map<Statement, Map<Variable, Variable>> defset) {
    CopyReplacer replacer = new CopyReplacer(defset);
    AstTraverser<Map<Variable, Variable>> traverser = new AstTraverser<Map<Variable, Variable>>(replacer);
    traverser.visit(ast, null);
    return replacer.getChanges();
  }

  public final int getChanges() {
    return changes + expr.getChanges();
  }

  @Override
  public Statement visit(Statement obj, Map<Variable, Variable> param) {
    return super.visit(obj, defset.get(obj));
  }

  /*
   * as we are already here, remove useless copies
   */
  @Override
  protected Statement visitAssignmentStmt(AssignmentStmt obj, Map<Variable, Variable> param) {
    obj.setSource(visit(obj.getSource(), param));
    if ((obj.getSource() instanceof VariableRefLinked) && (obj.getDestination() instanceof VariableRefLinked)) {
      VariableRefLinked src = (VariableRefLinked) obj.getSource();
      VariableRefLinked dst = (VariableRefLinked) obj.getDestination();

      if (src.getReference() == dst.getReference()) {
        changes++;
        return new NullStmt();
      }
    }
    return obj;
  }

  @Override
  public Expression visit(Expression expr, Map<Variable, Variable> param) {
    assert (param != null);
    return this.expr.visit(expr, param);
  }

  @Override
  public Variable visit(Variable expr, Map<Variable, Variable> param) {
    assert (param != null);
    return this.expr.visit(expr, param);
  }

}

class ExprCopyReplacer extends AstExpressionTraverser<Map<Variable, Variable>> {
  private int changes = 0;

  public final int getChanges() {
    return changes;
  }

  /*
   * here is the main action!
   */
  @Override
  protected Expression visitVariableRefLinked(VariableRefLinked obj, Map<Variable, Variable> param) {
    if (param.containsKey(obj.getReference())) {
      changes++;
      return new VariableRefLinked(param.get(obj.getReference()));
    } else {
      return obj;
    }
  }

}
