/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package graphviz;


import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Set;

import knowledge.KnowCallDst;
import knowledge.KnowOwner;
import knowledge.KnowledgeBase;

import org.apache.commons.lang.StringEscapeUtils;

import cfg.basicblock.BasicBlock;
import cfg.basicblock.BbEdge;
import cfg.function.Function;
import cfg.statement.Statement;
import cfg.variable.Variable;

public class FullLabel implements LabelWriter<BasicBlock, BbEdge> {

  static public final String SILVER       = "#cccccc";
  static public final String ORANGE       = "ORANGE";
  static public final String LIME         = "#9bff9b";
  static public final String GREEN        = "GREEN";
  static public final String RED          = "RED";
  static public final String WHITE        = "WHITE";
  static public final String COLOR_NORMAL = WHITE;
  static public final String COLOR_FUNC   = LIME;

  private String             prefix, sufix;
  private boolean            printDeleted;
  private KnowledgeBase    kb;
  private KnowOwner          ko;

  public FullLabel(String prefix, String sufix, boolean printDeleted, KnowledgeBase kb) {
    super();
    this.kb = kb;
    this.ko = (KnowOwner) kb.getEntry(KnowOwner.class);
    this.prefix = prefix;
    this.sufix = sufix;
    this.printDeleted = printDeleted;
  }

  public void writeVertexLabel(BasicBlock bb, PrintStream stream) {
    boolean isFuncEntry = ko.getBbOwner(bb).getEntry() == bb;
    stream.print("<TABLE BORDER=\"1\" CELLBORDER=\"0\" CELLSPACING=\"0\" ");
    stream.print(">");

    stream.print("<TR>");
    stream.print("<TD>");
    stream.print("</TD>");
    stream.print("<TD");
    if (isFuncEntry) {
      stream.print(" BGCOLOR=\"" + COLOR_FUNC + "\"");
    }
    stream.print(">");
    stream.print(bb);
    stream.print("</TD>");
    stream.print("</TR>");

    if (isFuncEntry) {
      stream.print("<TR>");
      stream.print("<TD>");
      stream.print("</TD>");
      stream.print("<TD>");
      writeArguments(ko.getBbOwner(bb).getInternParam(), stream);
      stream.print("</TD>");
      stream.print("</TR>");
    }

    // writeVariables( bb.use, COLOR_USE, stream );

    for (Statement inst : bb.getCode()) {
      // stmtWriter.visit( inst, stream );
      writeInstruction(inst, stream);
    }

    // writeVariables( bb.def, COLOR_DEF, stream );
    // writeVariables( bb.kill, COLOR_KILL, stream );

    stream.print("</TABLE>");
  }

  /*
   * private static void writeVariables(Collection<? extends Object> use, String color, PrintStream stream) { for(
   * Object var : use ){ stream.print( "<TR>" ); stream.print( "<TD BGCOLOR=\"" + color + "\">" ); stream.print( var );
   * stream.print( "</TD>" ); stream.print( "</TR>" ); } }
   */

  private void writeArguments(ArrayList<Variable> param, PrintStream stream) {
    for (Variable var : param) {
      stream.print(var);
      stream.print(" ");
    }
  }

  private void writeInstruction(Statement inst, PrintStream stream) {
    if (!inst.isDeleted() || printDeleted) {
      KnowCallDst callDst = (KnowCallDst) kb.getEntry(KnowCallDst.class);
      Set<Function> calldst = callDst.getCallDst(inst);
      stream.print("<TR>");
      stream.print("<TD ALIGN=\"RIGHT\" BGCOLOR=\"" + SILVER + "\" >");
      stream.print(inst.getNumber());
      stream.print("</TD>");

      stream.print("<TD ALIGN=\"LEFT\" BGCOLOR=\"" + getColor(inst) + "\" ");
      if (!calldst.isEmpty()) {
        stream.print("HREF=\"" + prefix + "." + calldst.iterator().next() + sufix + "\" ");
      }
      stream.print(" >");
      stream.print(StringEscapeUtils.escapeHtml(inst.toString()));
      stream.print("</TD>");
      stream.print("<TD ALIGN=\"RIGHT\">");
      // stream.print( inst.getReferencees().size() );
      stream.print("</TD>");
      stream.print("</TR>");
    }
  }

  private String getColor(Statement stmt) {
    if (stmt.isDeleted()) {
      return RED;
    } else {
      return COLOR_NORMAL;
    }
  }

  public void writeEdgeLabel(BbEdge e, PrintStream stream) {
    stream.print(e.contentString());
  }

}
