/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package phases;

import graphviz.DotHtmlWriter;
import graphviz.DotWriter;
import graphviz.FullLabel;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;

import cfg.Application;
import cfg.basicblock.BasicBlock;
import cfg.basicblock.BbEdge;
import cfg.function.PrgFunction;
import codewriter.BbWriter;
import codewriter.CodeWriter;
import codewriter.Writer;

public class Upcompiler {
  private LinkedList<Integer> nr      = new LinkedList<Integer>();
  private LinkedList<Long>    time    = new LinkedList<Long>();
  private int                 deepth  = 0;
  private Map<Integer, Long>  byNr    = new HashMap<Integer, Long>();
  private Map<String, Long>    byPhase = new HashMap<String, Long>();

  public void phase(Phase phase, Application app) {
    deepth++;
    if (deepth > nr.size()) {
      nr.push(0);
      time.push(0l);
    }
    assert (deepth == nr.size());

    time.pop();
    time.push(System.currentTimeMillis());

    phase.process(app);

    if (nr.size() > deepth) {
      nr.pop();
      time.pop();
    }
    assert (deepth == nr.size());
    long delta = System.currentTimeMillis() - time.peek();

    long old;
    if (byNr.containsKey(nr.getFirst())) {
      old = byNr.get(nr.getFirst());
    } else {
      old = 0;
    }
    old += delta;
    byNr.put(nr.getFirst(), old);

    if (byPhase.containsKey(phase.getClass().getName())) {
      old = byPhase.get(phase.getClass().getName());
    } else {
      old = 0;
    }
    old += delta;
    byPhase.put(phase.getClass().getName(), old);

    switch (phase.outArtefact()) {
      case SSA:
        printProgramLst(app, phase.getClass().getSimpleName());
        printFunctionGraph(app, phase.getClass().getSimpleName());
        break;
      case AST:
        writeProgram(app, phase.getClass().getSimpleName());
        break;
      case CODE: // nothing to do since the phase writes the files itself
        long id = System.currentTimeMillis();
        writeTime(byNr, app.getFile() + ".nr.csv", id);
        writeTime(byPhase, app.getFile() + ".phase.csv", id);
        break;
      default:
        throw new RuntimeException("Unhandled artefact: " + phase.outArtefact());
    }

    nr.push(nr.pop() + 1);
    deepth--;
  }

  private <T extends Comparable> void writeTime(Map<T, Long> map, String name, long id) {
    ArrayList<T> list = new ArrayList<T>(map.keySet());
    Collections.sort(list);

    try {
      PrintStream stream = new PrintStream(new FileOutputStream(name, true));
      
      stream.print("id");
      for (T itr : list) {
        stream.print(",");
        stream.print(itr);
      }
      stream.println();
      
      stream.print(id);
      for (T itr : list) {
        stream.print(",");
        stream.print(map.get(itr));
      }
      stream.println();

      stream.close();
    } catch (FileNotFoundException e) {
      e.printStackTrace();
    }
  }

  private void printProgramLst(Application app, String name) {
    name = app.getFile() + nrName() + "." + name + ".bb";
    BbWriter.write(app.getFunctions(), name);
  }

  private void printFunctionGraph(Application app, String name) {
    String prefix = app.getFile() + nrName() + "." + name;
    String sufix = ".gv";
    for (PrgFunction func : app.getFunctions()) {
      DotWriter<BasicBlock, BbEdge> writer = new DotHtmlWriter<BasicBlock, BbEdge>(new FullLabel(prefix,
          sufix + ".svg", false, app.getKb()), func.getGraph());
      writer.write(prefix + "." + func + sufix);
    }
  }

  private void writeProgram(Application app, String name) {
    name = app.getFile() + nrName() + "." + name + ".code";
    CodeWriter codewriter = new CodeWriter(null);
    PrintStream stream;
    try {
      stream = new PrintStream(name);
      Writer writer = new Writer(stream);
      codewriter.visit(app.getPrg(), writer);
      stream.close();
    } catch (FileNotFoundException e) {
      e.printStackTrace();
    }
  }

  private String nrName() {
    String result = "";
    for (int k : nr) {
      result = "." + k + result;
    }
    return result;
  }

}
