/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package reduction;


import java.util.Set;

import knowledge.KnowLinks;
import knowledge.KnowledgeBase;
import cfg.Condition;
import cfg.IrElement;
import cfg.IrStmtTraverser;
import cfg.basicblock.BasicBlock;
import cfg.expression.BooleanExpr;
import cfg.expression.CompareExpr;
import cfg.expression.IfExpr;
import cfg.statement.AssignmentStmt;
import cfg.statement.JumpStmt;
import cfg.statement.Statement;
import disassembler.diStorm3.DecomposedInst;

/**
 * Replaces the flag testing condition with an high level statement. Should be run exactly once over the CFG.
 * 
 * @author urs
 * 
 */
public class ConditionReduction extends IrStmtTraverser<Void, BasicBlock> {
  private KnowledgeBase     kb;
  private MemoryModelEnforcer mm;

  public ConditionReduction(KnowledgeBase kb, MemoryModelEnforcer mm) {
    super();
    this.kb = kb;
    this.mm = mm;
  }

  @Override
  protected Void visitBasicBlock(BasicBlock obj, BasicBlock param) {
    assert (param == null);
    return super.visitBasicBlock(obj, obj);
  }

  @Override
  protected Void visitAssignmentStmt(AssignmentStmt obj, BasicBlock param) {
    DecomposedInst orig = obj.getOriginal();
    if (orig == null) {
      return super.visitAssignmentStmt(obj, param);
    }
    Condition cond = Condition.parse(orig.getOpcode());
    if (cond == null) {
      return super.visitAssignmentStmt(obj, param);
    }

    switch (obj.getSource().getIrType()) {
      case IfExpr: {
        IfExpr ifex = (IfExpr) obj.getSource();

        Statement cmp = getReferencedStmt(ifex.getCondition());
        if (cmp != null) {
          if (!mm.isMoveAllowed(param.getCode(), cmp, obj)) {
            throw new RuntimeException("move of statement/expression not allowed");
          }
          BooleanExpr expr = ConditionParser.findExpression(cond, cmp);

          ifex.setCondition(expr);
        }
        break;
      }
      case CompareExpr: {
        CompareExpr cmex = (CompareExpr) obj.getSource();

        Statement cmp = getReferencedStmt(cmex);
        if (cmp != null) {
          if (!mm.isMoveAllowed(param.getCode(), cmp, obj)) {
            throw new RuntimeException("move of statement/expression not allowed");
          }
          BooleanExpr expr = ConditionParser.findExpression(cond, cmp);

          obj.setSource(expr);
        }
        break;
      }
      default: {
        throw new RuntimeException("Not yet implemented: " + obj.getSource());
      }
    }

    return null;
  }

  @Override
  protected Void visitJumpStmt(JumpStmt obj, BasicBlock param) {
    Statement cmp = getReferencedStmt(obj);
    if (cmp != null) {
      if (!mm.isMoveAllowed(param.getCode(), cmp, obj)) {
        throw new RuntimeException("move of statement/expression not allowed");
      }
      Condition cond = Condition.parse(obj.getOriginal().getOpcode());
      if (cond != null) { // probably a parsed jump table
        assert (cond != null);
        BooleanExpr expr = ConditionParser.findExpression(cond, cmp);

        obj.setExpression(expr);
      }
    }

    return null;
  }

  private Statement getReferencedStmt(IrElement obj) {
    KnowLinks kl = (KnowLinks) kb.getEntry(KnowLinks.class);
    Set<Statement> set = kl.getLinks(obj);
    switch (set.size()) {
      case 0:
        return null;
      case 1:
        return set.iterator().next();
      default:
        throw new RuntimeException("Don't know how to handle multiple references: " + obj);
    }
  }

}
