/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package reduction;


import java.util.Collection;

import cfg.Assignable;
import cfg.IntConstant;
import cfg.IrReplaceExprTraverser;
import cfg.expression.Expression;
import cfg.expression.IntegerExpr;
import cfg.function.PrgFunction;
import cfg.statement.AssignmentStmt;
import cfg.variable.VariablePtrDeref;
import cfg.variable.VariablePtrOf;

public class MathExprReduction extends IrReplaceExprTraverser<Void> {

  static public void reduceMath(Collection<PrgFunction> obj) {
    MathExprReduction mer = new MathExprReduction();
    mer.visitCollection(obj, null);
  }

  @Override
  protected void visitAssignmentStmt(AssignmentStmt obj, Void param) {
    super.visitAssignmentStmt(obj, param);
    for( int i = 0; i < obj.getDestination().size(); i++ ){
      Assignable dst = obj.getDestination().get(i);
      if( dst instanceof VariablePtrDeref ){
        dst = (Assignable) visitVariablePtrDeref( (VariablePtrDeref) dst, param );
        obj.getDestination().set(i, dst);
      }
    }
  }

  @Override
  protected Expression visitVariablePtrDeref(VariablePtrDeref obj, Void param) {
    Expression expr = visitExpression(obj.getExpression(), param);
    if (expr instanceof VariablePtrOf) {
      return ((VariablePtrOf) expr).getExpression();
    } else {
      obj.setExpression(expr);
      return obj;
    }
  }

  @Override
  protected Expression visitIntegerExpr(IntegerExpr obj, Void param) {
    obj = (IntegerExpr) super.visitIntegerExpr(obj, param);

    IntConstant left = null;
    IntConstant right = null;

    if (obj.getLeft() instanceof IntConstant) {
      left = (IntConstant) obj.getLeft();
    }
    if (obj.getRight() instanceof IntConstant) {
      right = (IntConstant) obj.getRight();
    }
    if ((left != null) && (right != null)) {
      return obj; // introduced by pointer to array transformation
    }
    if ((left == null) && (right == null)) {
      return obj;
    }

    IntConstant cons;
    Expression var;

    if (left != null) {
      cons = left;
      var = obj.getRight();
    } else {
      cons = right;
      var = obj.getLeft();
    }

    switch (obj.getOp()) {
      case Add:
      case Xor: {
        switch ((int) cons.getValue()) {
          case 0:
            return var;
          default:
            return obj;
        }
      }
      case And: {
        switch ((int) cons.getValue()) {
          case 0:
            return new IntConstant(0);
          default:
            return obj;
        }
      }
      case Div: {
        switch ((int) cons.getValue()) {
          case 1:
            return var; // should not occur
          default:
            return obj;
        }
      }
      case Mul: {
        switch ((int) cons.getValue()) {
          case 0:
            return new IntConstant(0);
          case 1:
            return var;
          default:
            return obj;
        }
      }
      case Or:
      case Sub:
      case ShiftArithmeticLeft:
      case ShiftArithmeticRight:
      case ShiftLeft:
      case ShiftRight: {
        return obj;
      }
      default: {
        throw new RuntimeException("Unhandled integer expression: " + obj.getOp());
      }
    }
  }
}
