/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package reduction;


import java.util.List;

import knowledge.KnowFuncCall;
import knowledge.KnowMemAccess;
import knowledge.MemAccess;
import cfg.Application;
import cfg.statement.Statement;

/**
 * Answers requests about movement of statements with respect to the memory model. It actually checks that memory writes
 * act as barriers for memory accesses.
 * 
 * The whole thing is quite crappy and should be replaced with something like VarDependencyGraph
 * 
 * @author urs
 * 
 */
public class MemoryModelEnforcer {

  private KnowMemAccess kma;
  private KnowFuncCall  kfc;

  public MemoryModelEnforcer(Application app) {
    super();
    kma = (KnowMemAccess) app.getKb().getEntry(KnowMemAccess.class);
    kfc = (KnowFuncCall) app.getKb().getEntry(KnowFuncCall.class);
  }

  public boolean isMoveAllowed(List<Statement> list, Statement src, Statement dst) {
    assert (!src.isDeleted());
    assert (!dst.isDeleted());

    if (kfc.hasFuncCall(src)) { // TODO check for pure function, those can be moved
      return false;
    } else if (kma.doesDynamicWrite(src)) {
      return checkList(list, src, dst, false);
    } else if (kma.doesDynamicRead(src)) {
      return checkList(list, src, dst, true);
    } else {
      List<MemAccess> statRead = kma.getStaticReads(src);
      List<MemAccess> statWrite = kma.getStaticWrites(src);

      if (statRead.isEmpty() && statWrite.isEmpty()) {
        // if memory is not involved, move is allowed anyway
        return true;
      }
      // only statically known memory addresses are involved

      if (statWrite.isEmpty()) {
        return checkStaticReadList(list, src, dst, statRead);
      }

      return checkList(list, src, dst, true); // TODO replace by more sophisticated method
      // return false;
    }
  }

  private boolean checkStaticReadList(List<Statement> list, Statement src, Statement dst, List<MemAccess> statRead) {
    for (int i = list.indexOf(src) + 1; i < list.size(); i++) {
      Statement itr = list.get(i);
      if (itr == dst) {
        return true;
      }
      if (kma.doesDynamicWrite(itr)) {
        return false;
      }
      List<MemAccess> statWrite = kma.getStaticWrites(itr);
      if (!statWrite.isEmpty()) {
        if (MemAccess.mayOverlap(statRead, statWrite)) {
          return false;
        }
      }
    }
    return false; // TODO: can we move it over BB boundaries?
  }

  private boolean checkList(List<Statement> list, Statement src, Statement dst, boolean onlyBlockOnWrite) {
    for (int i = list.indexOf(src) + 1; i < list.size(); i++) {
      Statement itr = list.get(i);
      if (itr == dst) {
        return true;
      }
      if (kma.doesWrite(itr)) {
        return false;
      }
      if (!onlyBlockOnWrite && kma.doesRead(itr)) {
        return false;
      }
    }
    return false; // TODO: can we move it over BB boundaries?
  }

}
