/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package reduction;


import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import cfg.Assignable;
import cfg.IntConstant;
import cfg.IrItr;
import cfg.IrReplaceExprTraverser;
import cfg.IrType;
import cfg.expression.Expression;
import cfg.expression.IntegerExpr;
import cfg.expression.IntegerOp;
import cfg.function.PrgFunction;
import cfg.function.argument.FuncVariables;
import cfg.matcher.VarIntopConstMatcher;
import cfg.statement.AssignmentStmt;
import cfg.unlinker.HllVariable;
import cfg.variable.Array;
import cfg.variable.ArrayAccess;
import cfg.variable.VariablePtrDeref;
import cfg.variable.VariablePtrOf;

public class PtrArrayReplacer extends IrReplaceExprTraverser<Void> {
  private final String     locVarPrefix = "loc";
  private final int        typesize     = 4;
  private PrgFunction      func         = null;
  private Map<Long, Array> arrayMap     = new HashMap<Long, Array>();

  static public void process(PrgFunction func, Set<Long> directOffsets) {
    PtrArrayReplacer replacer = new PtrArrayReplacer(func, directOffsets);
    replacer.visit(func, null);
  }

  public PtrArrayReplacer(PrgFunction func, Set<Long> directOffsets) {
    this.func = func;
    createArrayInfo(directOffsets);
  }

  private void createArrayInfo(Set<Long> directOffsets) {
    ArrayList<Long> offsets = new ArrayList<Long>(directOffsets);
    Collections.sort(offsets);

    if (!offsets.isEmpty()) {
      assert (offsets.get(offsets.size() - 1) < func.getStacksize());
      offsets.add((long) func.getStacksize());

      for (int i = 0; i < offsets.size(); i++) {
        long val = offsets.get(i);
        assert (val % 4 == 0);
        offsets.set(i, val / 4);
      }

      ArrayList<Long> size = new ArrayList<Long>(offsets.size() - 1);
      for (int i = 0; i < offsets.size() - 1; i++) {
        long val = offsets.get(i + 1) - offsets.get(i);
        size.add(val);
      }
      offsets.remove(offsets.size() - 1);
      createLocalArrays(offsets, size);
    }
  }

  private void createLocalArrays(ArrayList<Long> offsets, ArrayList<Long> size) {
    assert (offsets.size() == size.size());

    for (int i = 0; i < offsets.size(); i++) {
      long num = offsets.get(i);
      Array var = new Array(new HllVariable(locVarPrefix + num), num, size.get(i));
      arrayMap.put(num, var);
      func.addArray(var);
    }

  }

  @Override
  protected void visitAssignmentStmt(AssignmentStmt obj, Void param) {
    List<Assignable> lhs = obj.getDestination();
    for (int i = 0; i < lhs.size(); i++) {
      if (lhs.get(i).getIrType() == IrType.VariablePtrDeref) {
        lhs.set(i, (Assignable) visitVariablePtrDeref((VariablePtrDeref) lhs.get(i), null));
      }
    }
    super.visitAssignmentStmt(obj, null);
  }

  @Override
  protected Expression visitVariablePtrDeref(VariablePtrDeref obj, Void param) {
    VarIntopConstMatcher matcher = new VarIntopConstMatcher();
    matcher.parse(new IrItr(obj.getExpression()));
    if (!matcher.hasError()) {
      if (matcher.getVar() == FuncVariables.locPtr) {
        if (matcher.getOp() != IntegerOp.Add) {
          throw new RuntimeException("Expect addition");
        }
        long index = matcher.getValue();
        assert (index % typesize == 0);
        index = index / typesize;

        Array arr = arrayMap.get(index);
        assert (arr != null);
        ArrayAccess var = new ArrayAccess(arr, new IntConstant(0));
        // VariableArray var = new VariableArray(new VariableRefLinked(func.getLocarray()), index);

        return var;
      }
    }
    return super.visitVariablePtrDeref(obj, param);
  }

  @Override
  protected Expression visitIntegerExpr(IntegerExpr obj, Void param) {
    VarIntopConstMatcher matcher = new VarIntopConstMatcher();
    matcher.parse(new IrItr(obj));
    if (!matcher.hasError()) {
      if (matcher.getVar() == FuncVariables.locPtr) {
        if (matcher.getOp() != IntegerOp.Add) {
          throw new RuntimeException("Expect addition");
        }
        long index = matcher.getValue();
        assert (index % typesize == 0);
        index = index / typesize;

        Array arr = arrayMap.get(index);
        assert (arr != null);
        ArrayAccess var = new ArrayAccess(arr, new IntConstant(0));
        // VariableArray var = new VariableArray(new VariableRefLinked(func.getLocarray()), index);

        return new VariablePtrOf(var);
      }
    }
    return super.visitIntegerExpr(obj, param);
  }

}
