/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package reduction;


import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import knowledge.KnowDeletable;
import knowledge.KnowLinks;
import knowledge.KnowledgeBase;
import cfg.IrType;
import cfg.basicblock.BasicBlock;
import cfg.function.Function;
import cfg.function.PrgFunction;
import cfg.statement.Statement;

/**
 * Marks unused statements as deleted
 * 
 * @author urs
 * 
 */
public class StmtDeleter {
  private KnowLinks      knowLinks;
  private KnowDeletable  knowDeletable;
  private Set<Statement> visited = new HashSet<Statement>();

  public static int process(PrgFunction func, KnowledgeBase kb) {
    StmtDeleter deleter = new StmtDeleter(kb);
    return deleter.tuwas(func);
  }

  public static int process(Collection<? extends Function> values, KnowledgeBase base) {
    int num = 0;
    for (Function func : values) {
      if (func.getIrType() == IrType.FuncProgram) {
        num += process((PrgFunction) func, base);
      }
    }
    return num;
  }

  public StmtDeleter(KnowledgeBase kb) {
    super();
    knowLinks = (KnowLinks) kb.getEntry(KnowLinks.class);
    knowDeletable = (KnowDeletable) kb.getEntry(KnowDeletable.class);
  }

  private int tuwas(PrgFunction func) {
    keep(func);
    return remove(func);
  }

  private int remove(PrgFunction func) {
    int num = 0;
    for (BasicBlock bb : func.getGraph().vertexSet()) {
      for (Statement stmt : bb.getCode()) {
        if (!visited.contains(stmt)) {
          if (!stmt.isDeleted()) {
            num++;
          }
          stmt.setDeleted();
        }
      }
    }
    return num;
  }

  private void keep(PrgFunction func) {
    for (BasicBlock bb : func.getGraph().vertexSet()) {
      for (Statement stmt : bb.getCode()) {
        if (!stmt.isDeleted()) {
          if (!knowDeletable.isDeletable(stmt)) {
            keep(stmt);
          }
        }
      }
    }
  }

  private void keep(Statement stmt) {
    if (!visited.contains(stmt)) {
      visited.add(stmt);
      for (Statement itr : knowLinks.getLinks(stmt)) {
        keep(itr);
      }
    }
  }

}
